<?php
require_once '../config.php';
startSession();

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $conn = getDbConnection();

    if ($action === 'register') {
        $name = sanitize($_POST['name'] ?? '');
        $email = sanitize($_POST['email'] ?? '');
        $phone = sanitize($_POST['phone'] ?? '');
        $password = $_POST['password'] ?? '';

        if (empty($name) || empty($email) || empty($password)) {
            json_response(['success' => false, 'message' => 'All fields are required'], 400);
        }

        // Check if email exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        if ($stmt->get_result()->num_rows > 0) {
            json_response(['success' => false, 'message' => 'Email already exists'], 400);
        }
        $stmt->close();

        // Create user
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $conn->prepare("INSERT INTO users (name, email, phone, password) VALUES (?, ?, ?, ?)");
        $stmt->bind_param("ssss", $name, $email, $phone, $hashedPassword);

        if ($stmt->execute()) {
            $userId = $stmt->insert_id;
            $_SESSION['user_id'] = $userId;
            $_SESSION['user_name'] = $name;
            $_SESSION['user_email'] = $email;
            
            json_response(['success' => true, 'message' => 'Registration successful', 'user_id' => $userId]);
        } else {
            json_response(['success' => false, 'message' => 'Registration failed'], 500);
        }
        $stmt->close();

    } elseif ($action === 'login') {
        $email = sanitize($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';

        if (empty($email) || empty($password)) {
            json_response(['success' => false, 'message' => 'Email and password are required'], 400);
        }

        $stmt = $conn->prepare("SELECT * FROM users WHERE email = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows > 0) {
            $user = $result->fetch_assoc();
            if (password_verify($password, $user['password'])) {
                session_regenerate_id(true);
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['user_name'] = $user['name'];
                $_SESSION['user_email'] = $user['email'];
                
                json_response(['success' => true, 'message' => 'Login successful', 'user' => [
                    'id' => $user['id'],
                    'name' => $user['name'],
                    'email' => $user['email'],
                    'phone' => $user['phone'] ?? null,
                    'profile_image' => $user['profile_image'] ?? null
                ]]);
            } else {
                json_response(['success' => false, 'message' => 'Invalid email or password'], 401);
            }
        } else {
            json_response(['success' => false, 'message' => 'Invalid email or password'], 401);
        }
        $stmt->close();

    } elseif ($action === 'logout') {
        session_destroy();
        json_response(['success' => true, 'message' => 'Logged out successfully']);

    } elseif ($action === 'check') {
        if (isUserLoggedIn()) {
            // Get user data including profile_image
            $stmt = $conn->prepare("SELECT id, name, email, phone, profile_image FROM users WHERE id = ?");
            $stmt->bind_param("i", $_SESSION['user_id']);
            $stmt->execute();
            $result = $stmt->get_result();
            $user = $result->fetch_assoc();
            $stmt->close();
            
            json_response(['success' => true, 'logged_in' => true, 'user' => [
                'id' => $user['id'],
                'name' => $user['name'],
                'email' => $user['email'],
                'phone' => $user['phone'] ?? null,
                'profile_image' => $user['profile_image'] ?? null
            ]]);
        } else {
            json_response(['success' => true, 'logged_in' => false]);
        }
        
    }

    $conn->close();
} else {
    json_response(['error' => 'Invalid request method'], 405);
}
?>

