<?php
require_once '../config.php';
startSession();

if (!isAdminLoggedIn()) {
    json_response(['error' => 'Unauthorized'], 401);
}

$conn = getDbConnection();

// Define upload directory (admin/api/uploads/)
$uploadDir = __DIR__ . '/uploads/';
if (!file_exists($uploadDir)) {
    if (!mkdir($uploadDir, 0755, true)) {
        json_response(['error' => 'Failed to create upload directory'], 500);
    }
    // Create .htaccess for public access
    $htaccessContent = "# Allow image access\n";
    $htaccessContent .= "Options -Indexes\n";
    $htaccessContent .= "<FilesMatch \"\.(jpg|jpeg|png|gif|webp)$\">\n";
    $htaccessContent .= "    Order Allow,Deny\n";
    $htaccessContent .= "    Allow from all\n";
    $htaccessContent .= "    Require all granted\n";
    $htaccessContent .= "</FilesMatch>\n";
    file_put_contents($uploadDir . '.htaccess', $htaccessContent);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
    $categoryId = isset($_POST['category_id']) ? (int)$_POST['category_id'] : 0;
    $title = sanitize($_POST['title'] ?? '');
    $description = sanitize($_POST['description'] ?? '');
    $downloadUrl = sanitize($_POST['download_url'] ?? '');
    $price = isset($_POST['price']) ? (float)$_POST['price'] : 0;
    // Get product type from dropdown (free/paid)
    $productType = sanitize($_POST['product_type'] ?? 'paid');
    $isFree = ($productType === 'free') ? 1 : 0;

    if (empty($title)) {
        json_response(['error' => 'Title is required'], 400);
    }

    // Handle image upload
    $previewImage = sanitize($_POST['preview_image'] ?? ''); // Keep existing if no new upload
    
    if (isset($_FILES['preview_image_upload']) && $_FILES['preview_image_upload']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['preview_image_upload'];
        
        // Validate file type
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/jpg'];
        $fileType = mime_content_type($file['tmp_name']);
        $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        if (!in_array($fileType, $allowedTypes) && !in_array($fileExtension, ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
            json_response(['error' => 'Invalid image type. Allowed: JPEG, PNG, GIF, WebP'], 400);
        }
        
        // Validate file size (5MB)
        if ($file['size'] > 5 * 1024 * 1024) {
            json_response(['error' => 'Image size too large. Maximum 5MB allowed.'], 400);
        }
        
        // Generate unique filename
        $filename = 'product_' . time() . '_' . uniqid() . '.' . $fileExtension;
        $filePath = $uploadDir . $filename;
        
        // Ensure directory exists and is writable
        if (!is_writable($uploadDir)) {
            @chmod($uploadDir, 0755);
        }
        
        // Move uploaded file
        if (move_uploaded_file($file['tmp_name'], $filePath)) {
            // Set proper permissions
            @chmod($filePath, 0644);
            
            // Delete old image if exists (for update)
            if ($id > 0 && !empty($previewImage) && file_exists($uploadDir . $previewImage)) {
                @unlink($uploadDir . $previewImage);
            }
            $previewImage = $filename;
        } else {
            $errorMsg = 'Failed to upload image';
            $lastError = error_get_last();
            if ($lastError) {
                $errorMsg .= ': ' . $lastError['message'];
            }
            $errorMsg .= '. Directory: ' . $uploadDir . ' (exists: ' . (file_exists($uploadDir) ? 'yes' : 'no') . ', writable: ' . (is_writable($uploadDir) ? 'yes' : 'no') . ')';
            json_response(['error' => $errorMsg], 500);
        }
    } elseif ($id > 0 && empty($previewImage)) {
        // If editing and no new upload, keep existing image from database
        $stmt = $conn->prepare("SELECT preview_image FROM products WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($row = $result->fetch_assoc()) {
            $previewImage = $row['preview_image'] ?? '';
        }
        $stmt->close();
    }

    if ($id > 0) {
        // Update
        $stmt = $conn->prepare("UPDATE products SET category_id = ?, title = ?, description = ?, preview_image = ?, download_url = ?, price = ?, is_free = ? WHERE id = ?");
        $stmt->bind_param("issssdii", $categoryId, $title, $description, $previewImage, $downloadUrl, $price, $isFree, $id);
    } else {
        // Insert
        $stmt = $conn->prepare("INSERT INTO products (category_id, title, description, preview_image, download_url, price, is_free) VALUES (?, ?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("issssdi", $categoryId, $title, $description, $previewImage, $downloadUrl, $price, $isFree);
    }

    if ($stmt->execute()) {
        header('Location: ../products.php?success=1');
        exit;
    } else {
        json_response(['error' => 'Failed to save product'], 500);
    }

    $stmt->close();
}

$conn->close();
?>

