<?php
require_once '../config.php';
startSession();

// Allow all origins (for mobile app)
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Accept');

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Set proper headers to avoid 406 errors
header('Content-Type: application/json; charset=utf-8');
header('Accept: application/json');

// Only process POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    json_response(['success' => false, 'message' => 'Only POST method allowed'], 405);
}

$conn = getDbConnection();

// Authentication: Prioritize user_id from POST (works better with multipart)
// Fall back to session if available
$userId = null;

// Check POST data first (multipart/form-data sends data here)
if (isset($_POST['user_id']) && !empty($_POST['user_id'])) {
    $userId = (int)$_POST['user_id'];
} elseif (isUserLoggedIn()) {
    // Fallback to session
    $userId = $_SESSION['user_id'];
}

// Validate user ID
if ($userId === null || $userId <= 0) {
    json_response([
        'success' => false,
        'message' => 'Unauthorized. User ID required.',
        'debug' => [
            'post_keys' => array_keys($_POST ?? []),
            'post_user_id' => $_POST['user_id'] ?? null,
            'has_session' => isUserLoggedIn(),
            'session_id' => $_SESSION['user_id'] ?? null,
        ]
    ], 401);
}

// Verify user exists in database
$stmt = $conn->prepare("SELECT id FROM users WHERE id = ?");
$stmt->bind_param("i", $userId);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    $stmt->close();
    json_response([
        'success' => false,
        'message' => 'Invalid user ID. User not found.',
    ], 401);
}
$stmt->close();

// Define upload directory (api/uploads/users/) - User images folder
// Go up from admin/api/ to root, then to api/uploads/users/
$uploadDir = dirname(dirname(__DIR__)) . '/api/uploads/users/';
if (!file_exists($uploadDir)) {
    if (!mkdir($uploadDir, 0755, true)) {
        json_response([
            'success' => false, 
            'message' => 'Failed to create upload directory: ' . $uploadDir,
            'debug' => ['dir' => $uploadDir, 'exists' => file_exists($uploadDir)]
        ], 500);
    }
    // Create .htaccess for public access
    $htaccessContent = "# Allow image access\n";
    $htaccessContent .= "Options -Indexes\n";
    $htaccessContent .= "<FilesMatch \"\.(jpg|jpeg|png|gif|webp)$\">\n";
    $htaccessContent .= "    Order Allow,Deny\n";
    $htaccessContent .= "    Allow from all\n";
    $htaccessContent .= "    Require all granted\n";
    $htaccessContent .= "</FilesMatch>\n";
    file_put_contents($uploadDir . '.htaccess', $htaccessContent);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Handle both regular POST and multipart/form-data
    $name = '';
    $phone = '';
    
    // Debug: Log received data
    $debugInfo = [
        'post_keys' => array_keys($_POST ?? []),
        'post_name' => $_POST['name'] ?? null,
        'post_phone' => $_POST['phone'] ?? null,
        'post_user_id' => $_POST['user_id'] ?? null,
        'has_files' => isset($_FILES['profile_image']),
        'file_error' => isset($_FILES['profile_image']) ? $_FILES['profile_image']['error'] : null,
    ];
    
    if (isset($_POST['name'])) {
        $name = sanitize($_POST['name']);
    }
    if (isset($_POST['phone'])) {
        $phone = sanitize($_POST['phone']);
    }
    
    // Validate name
    if (empty($name)) {
        json_response([
            'success' => false, 
            'message' => 'Name is required',
            'debug' => $debugInfo
        ], 400);
    }
    
    // Convert empty phone to NULL for database
    // Keep as empty string in variable, but treat as NULL in SQL
    $phoneValue = (empty($phone)) ? null : $phone;
    
    // Handle profile image upload
    $profileImage = null;
    
    if (isset($_FILES['profile_image']) && $_FILES['profile_image']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['profile_image'];
        
        // Validate file type
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/jpg'];
        $fileType = mime_content_type($file['tmp_name']);
        $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        if (!in_array($fileType, $allowedTypes) && !in_array($fileExtension, ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
            json_response(['success' => false, 'message' => 'Invalid image type. Allowed: JPEG, PNG, GIF, WebP'], 400);
        }
        
        // Validate file size (2MB)
        if ($file['size'] > 2 * 1024 * 1024) {
            json_response(['success' => false, 'message' => 'Image size too large. Maximum 2MB allowed.'], 400);
        }
        
        // Generate unique filename
        $filename = 'profile_' . $userId . '_' . time() . '_' . uniqid() . '.' . $fileExtension;
        $filePath = $uploadDir . $filename;
        
        // Ensure directory is writable
        if (!is_writable($uploadDir)) {
            @chmod($uploadDir, 0755);
        }
        
        // Move uploaded file
        if (move_uploaded_file($file['tmp_name'], $filePath)) {
            // Set proper permissions
            @chmod($filePath, 0644);
            
            // Get old profile image to delete later
            $stmt = $conn->prepare("SELECT profile_image FROM users WHERE id = ?");
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            $oldImage = null;
            if ($row = $result->fetch_assoc()) {
                $oldImage = $row['profile_image'];
            }
            $stmt->close();
            
            // Verify file was actually moved
            if (!file_exists($filePath)) {
                json_response([
                    'success' => false, 
                    'message' => 'File upload failed - file not found after move',
                    'debug' => ['target_path' => $filePath, 'tmp_name' => $file['tmp_name']]
                ], 500);
            }
            
            // Delete old image if exists (check both old and new paths for migration)
            if ($oldImage && !empty($oldImage)) {
                // Check old path (admin/api/uploads/profiles/)
                $oldPath1 = dirname(dirname(__DIR__)) . '/admin/api/uploads/profiles/' . $oldImage;
                if (file_exists($oldPath1)) {
                    @unlink($oldPath1);
                }
                // Check new path (api/uploads/users/)
                $oldPath2 = $uploadDir . $oldImage;
                if (file_exists($oldPath2)) {
                    @unlink($oldPath2);
                }
            }
            
            $profileImage = $filename;
        } else {
            $lastError = error_get_last();
            json_response([
                'success' => false, 
                'message' => 'Failed to upload image. Error: ' . ($lastError ? $lastError['message'] : 'Unknown error'),
                'debug' => [
                    'tmp_name' => $file['tmp_name'],
                    'target_path' => $filePath,
                    'upload_dir' => $uploadDir,
                    'dir_exists' => file_exists($uploadDir),
                    'dir_writable' => is_writable($uploadDir),
                    'error_code' => $file['error']
                ]
            ], 500);
        }
    }
    
    // Update user profile
    // Build UPDATE query dynamically
    $updateFields = ['name = ?'];
    $bindParams = [$name];
    $bindTypes = 's';
    
    // Handle phone properly - use NULL in SQL if empty, otherwise bind parameter
    if ($phoneValue === null || $phone === '') {
        $updateFields[] = 'phone = NULL';
    } else {
        $updateFields[] = 'phone = ?';
        $bindParams[] = $phoneValue;
        $bindTypes .= 's';
    }
    
    // Add profile image if uploaded
    if ($profileImage !== null) {
        $updateFields[] = 'profile_image = ?';
        $bindParams[] = $profileImage;
        $bindTypes .= 's';
    }
    
    // Add user ID for WHERE clause
    $bindParams[] = $userId;
    $bindTypes .= 'i';
    
    $sql = "UPDATE users SET " . implode(', ', $updateFields) . " WHERE id = ?";
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        json_response([
            'success' => false, 
            'message' => 'Database prepare error: ' . $conn->error
        ], 500);
    }
    
    $stmt->bind_param($bindTypes, ...$bindParams);
    
    if ($stmt->execute()) {
        // Check if any rows were actually updated
        $affectedRows = $stmt->affected_rows;
        
        // Get updated user data
        $stmt2 = $conn->prepare("SELECT name, email, phone, profile_image FROM users WHERE id = ?");
        $stmt2->bind_param("i", $userId);
        $stmt2->execute();
        $result = $stmt2->get_result();
        $user = $result->fetch_assoc();
        $stmt2->close();
        
        if ($user) {
            // Update session
            $_SESSION['user_name'] = $user['name'];
            
            json_response([
                'success' => true, 
                'message' => 'Profile updated successfully',
                'affected_rows' => $affectedRows,
                'user' => [
                    'id' => $userId,
                    'name' => $user['name'],
                    'email' => $user['email'],
                    'phone' => $user['phone'],
                    'profile_image' => $user['profile_image']
                ]
            ]);
        } else {
            json_response([
                'success' => false, 
                'message' => 'User not found after update'
            ], 404);
        }
    } else {
        json_response([
            'success' => false, 
            'message' => 'Failed to update profile. SQL Error: ' . $stmt->error . ' | DB Error: ' . $conn->error,
            'sql' => $sql,
            'bind_types' => $bindTypes,
            'bind_params_count' => count($bindParams)
        ], 500);
    }
    
    $stmt->close();
} else {
    json_response(['error' => 'Invalid request method'], 405);
}

$conn->close();
?>

