<?php
require_once 'config.php';
startSession();

if (!isAdminLoggedIn()) {
    header('Location: login.php');
    exit;
}

$conn = getDbConnection();
$message = '';
$messageType = '';

// Handle add/edit
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = $conn->real_escape_string($_POST['name'] ?? '');
    $slug = $conn->real_escape_string($_POST['slug'] ?? '');
    $icon = $conn->real_escape_string($_POST['icon'] ?? 'script');
    
    if (empty($name)) {
        $message = 'Category name is required';
        $messageType = 'error';
    } else {
        if (isset($_POST['id']) && !empty($_POST['id'])) {
            // Edit
            $id = (int)$_POST['id'];
            $stmt = $conn->prepare("UPDATE categories SET name = ?, slug = ?, icon = ? WHERE id = ?");
            $stmt->bind_param("sssi", $name, $slug, $icon, $id);
            if ($stmt->execute()) {
                $message = 'Category updated successfully';
                $messageType = 'success';
            } else {
                $message = 'Failed to update category';
                $messageType = 'error';
            }
            $stmt->close();
        } else {
            // Add
            $stmt = $conn->prepare("INSERT INTO categories (name, slug, icon) VALUES (?, ?, ?)");
            $stmt->bind_param("sss", $name, $slug, $icon);
            if ($stmt->execute()) {
                $message = 'Category added successfully';
                $messageType = 'success';
            } else {
                $message = 'Failed to add category';
                $messageType = 'error';
            }
            $stmt->close();
        }
    }
}

// Handle delete
if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    $stmt = $conn->prepare("DELETE FROM categories WHERE id = ?");
    $stmt->bind_param("i", $id);
    if ($stmt->execute()) {
        $message = 'Category deleted successfully';
        $messageType = 'success';
    } else {
        $message = 'Failed to delete category';
        $messageType = 'error';
    }
    $stmt->close();
}

// Get category for editing
$editCategory = null;
if (isset($_GET['edit'])) {
    $id = (int)$_GET['edit'];
    $result = $conn->query("SELECT * FROM categories WHERE id = $id");
    if ($result->num_rows > 0) {
        $editCategory = $result->fetch_assoc();
    }
}

// Get all categories
$categories = $conn->query("SELECT * FROM categories ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Categories - Admin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:opsz,wght,FILL,GRAD@20..48,100..700,0..1,-50..200" />
    <link rel="stylesheet" href="https://fonts.googleapis.com/icon?family=Material+Icons">
    <style>
        .material-icons {
            font-family: 'Material Icons';
            font-weight: normal;
            font-style: normal;
            font-size: 24px;
            line-height: 1;
            letter-spacing: normal;
            text-transform: none;
            display: inline-block;
            white-space: nowrap;
            word-wrap: normal;
            direction: ltr;
            -webkit-font-feature-settings: 'liga';
            -webkit-font-smoothing: antialiased;
        }
    </style>
</head>
<body class="bg-gray-100">
    <?php include 'sidebar.php'; ?>
    
    <div class="ml-64 p-6">
        <div class="flex justify-between items-center mb-6">
            <h2 class="text-3xl font-bold text-gray-800">Categories Management</h2>
            <button onclick="openAddModal()" class="bg-teal-600 text-white px-4 py-2 rounded-lg hover:bg-teal-700 flex items-center">
                <span class="material-icons mr-2" style="font-size: 18px;">add</span> Add Category
            </button>
        </div>

        <?php if ($message): ?>
            <div class="mb-4 p-4 rounded-lg <?php echo $messageType === 'success' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

<div class="bg-white rounded-lg shadow-lg overflow-hidden">
    <div class="overflow-x-auto">
        <table class="min-w-full divide-y divide-gray-200 min-w-[640px]">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Icon</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Name</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Slug</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php if (empty($categories)): ?>
                        <tr>
                            <td colspan="4" class="px-6 py-4 text-center text-gray-500">No categories found</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($categories as $cat): ?>
                            <tr>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <?php
                                    // Get Material icon for display
                                    $displayIcon = $cat['icon'] ?? 'script';
                                    $iconMap = [
                                        'script' => 'insert_drive_file',
                                        'code' => 'code',
                                        'laravel' => 'developer_mode',
                                        'wordpress' => 'web',
                                        'wordpress-themes' => 'palette',
                                        'plugins' => 'extension',
                                        'template' => 'description',
                                        'addon' => 'add_circle',
                                        'mobile-apps' => 'phone_android',
                                        'html-template' => 'code',
                                        'php' => 'developer_mode',
                                        'blogger' => 'article',
                                        'nodejs' => 'storage',
                                        'flutter' => 'phone_android'
                                    ];
                                    $materialIcon = $iconMap[$displayIcon] ?? ($iconMap[str_replace('fa-', '', $displayIcon)] ?? 'insert_drive_file');
                                    ?>
                                    <span class="material-icons text-2xl text-teal-600"><?php echo htmlspecialchars($materialIcon); ?></span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                    <?php echo htmlspecialchars($cat['name']); ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                    <?php echo htmlspecialchars($cat['slug'] ?? '-'); ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                    <a href="?edit=<?php echo $cat['id']; ?>" class="text-teal-600 hover:text-teal-900 mr-3 inline-flex items-center">
                                        <span class="material-icons" style="font-size: 18px; margin-right: 4px;">edit</span> Edit
                                    </a>
                                    <a href="?delete=<?php echo $cat['id']; ?>" 
                                       onclick="return confirm('Are you sure you want to delete this category?')"
                                       class="text-red-600 hover:text-red-900 inline-flex items-center">
                                        <span class="material-icons" style="font-size: 18px; margin-right: 4px;">delete</span> Delete
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Add/Edit Modal -->
    <div id="categoryModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center">
        <div class="bg-white rounded-lg p-6 w-full max-w-md">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-xl font-bold"><?php echo $editCategory ? 'Edit' : 'Add'; ?> Category</h3>
                <button onclick="closeModal()" class="text-gray-500 hover:text-gray-700">
                    <span class="material-icons">close</span>
                </button>
            </div>
            
            <form method="POST" action="">
                <?php if ($editCategory): ?>
                    <input type="hidden" name="id" value="<?php echo $editCategory['id']; ?>">
                <?php endif; ?>
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Category Name *</label>
                    <input type="text" name="name" required
                           value="<?php echo htmlspecialchars($editCategory['name'] ?? ''); ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-teal-500">
                </div>

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Slug</label>
                    <input type="text" name="slug"
                           value="<?php echo htmlspecialchars($editCategory['slug'] ?? ''); ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-teal-500"
                           placeholder="category-slug">
                </div>

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Icon <span class="text-red-500">*</span>
                    </label>
                    <select name="icon" id="iconInput" required
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-teal-500">
                        <?php
                        // Predefined icons list (matching Material icons in Flutter app)
                        $predefinedIcons = [
                            'script' => ['label' => 'Script', 'material' => 'insert_drive_file'],
                            'code' => ['label' => 'Code', 'material' => 'code'],
                            'laravel' => ['label' => 'Laravel', 'material' => 'developer_mode'],
                            'wordpress' => ['label' => 'WordPress', 'material' => 'web'],
                            'wordpress-themes' => ['label' => 'WordPress Themes', 'material' => 'palette'],
                            'plugins' => ['label' => 'Plugins', 'material' => 'extension'],
                            'template' => ['label' => 'Template', 'material' => 'description'],
                            'addon' => ['label' => 'Addon', 'material' => 'add_circle'],
                            'mobile-apps' => ['label' => 'Mobile Apps', 'material' => 'phone_android'],
                            'html-template' => ['label' => 'HTML Template', 'material' => 'code'],
                            'php' => ['label' => 'PHP', 'material' => 'developer_mode'],
                            'blogger' => ['label' => 'Blogger', 'material' => 'article'],
                            'nodejs' => ['label' => 'Node.js', 'material' => 'storage'],
                            'flutter' => ['label' => 'Flutter', 'material' => 'phone_android'],
                        ];
                        
                        $currentIcon = $editCategory['icon'] ?? 'script';
                        // Remove 'fa-' prefix if present for comparison
                        $cleanCurrentIcon = $currentIcon;
                        if (strpos($cleanCurrentIcon, 'fa-') === 0) {
                            $cleanCurrentIcon = substr($cleanCurrentIcon, 3);
                        }
                        
                        foreach ($predefinedIcons as $iconValue => $iconData):
                            $iconLabel = $iconData['label'];
                            $iconFaClass = $iconData['fa'];
                            // Store clean icon name (without fa- prefix) for Flutter Material icons
                            $selected = ($cleanCurrentIcon === $iconValue) ? 'selected' : '';
                        ?>
                            <option value="<?php echo htmlspecialchars($iconValue); ?>" <?php echo $selected; ?>>
                                <?php echo htmlspecialchars($iconLabel); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <p class="text-xs text-gray-500 mt-1 mb-2">
                        💡 Select an icon from the predefined list above
                    </p>
                    <div class="mt-2 p-3 bg-gray-50 rounded-md">
                        <p class="text-xs text-gray-600 mb-2">Preview:</p>
                        <div class="text-center">
                            <span id="iconPreview" class="material-icons text-3xl text-teal-600"></span>
                            <p class="text-xs text-gray-500 mt-1" id="iconNamePreview">
                                <?php 
                                $previewIcon = $editCategory['icon'] ?? 'script';
                                $previewName = str_replace('fa-', '', $previewIcon);
                                echo ucfirst(str_replace('-', ' ', $previewName));
                                ?>
                            </p>
                        </div>
                    </div>
                </div>

                <div class="flex justify-end space-x-3">
                    <button type="button" onclick="closeModal()"
                            class="px-4 py-2 border border-gray-300 rounded-md hover:bg-gray-50">
                        Cancel
                    </button>
                    <button type="submit"
                            class="px-4 py-2 bg-teal-600 text-white rounded-md hover:bg-teal-700">
                        <?php echo $editCategory ? 'Update' : 'Add'; ?> Category
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        const iconInput = document.getElementById('iconInput');
        const iconPreview = document.getElementById('iconPreview');
        const iconNamePreview = document.getElementById('iconNamePreview');

        function openAddModal() {
            document.getElementById('categoryModal').classList.remove('hidden');
            if (iconInput) {
                iconInput.value = 'script';
                updateIconPreview();
            }
        }

        function closeModal() {
            document.getElementById('categoryModal').classList.add('hidden');
            window.location.href = 'categories.php';
        }

        // Icon mapping for preview (Material icons)
        const iconMap = {
            'script': 'insert_drive_file',
            'code': 'code',
            'laravel': 'developer_mode',
            'wordpress': 'web',
            'wordpress-themes': 'palette',
            'plugins': 'extension',
            'template': 'description',
            'addon': 'add_circle',
            'mobile-apps': 'phone_android',
            'html-template': 'code',
            'php': 'developer_mode',
            'blogger': 'article',
            'nodejs': 'storage',
            'flutter': 'phone_android'
        };
        
        function updateIconPreview() {
            if (!iconInput || !iconPreview || !iconNamePreview) return;
            
            const iconValue = iconInput.value.trim() || 'script';
            // Get Material icon name for preview
            const materialIcon = iconMap[iconValue] || 'insert_drive_file';
            
            // Format icon name for display
            const iconName = iconValue.split('-').map(word => 
                word.charAt(0).toUpperCase() + word.slice(1)
            ).join(' ');
            
            iconPreview.className = 'material-icons text-3xl text-teal-600';
            iconPreview.textContent = materialIcon;
            iconNamePreview.textContent = iconName || 'Script';
        }

        if (iconInput) {
            iconInput.addEventListener('change', updateIconPreview);
            
            <?php if ($editCategory): ?>
                updateIconPreview();
                document.getElementById('categoryModal').classList.remove('hidden');
            <?php endif; ?>
        }
    </script>

<?php include 'layout_end.php'; ?>

