<?php
require_once 'config.php';
startSession();

if (!isAdminLoggedIn()) {
    header('Location: login.php');
    exit;
}

$conn = getDbConnection();

// Handle create notification
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'create') {
    $title = sanitize($_POST['title'] ?? '');
    $message = sanitize($_POST['message'] ?? '');
    $type = sanitize($_POST['type'] ?? 'info');
    $isActive = isset($_POST['is_active']) ? 1 : 0;
    
    if (!empty($title) && !empty($message)) {
        $stmt = $conn->prepare("INSERT INTO notifications (title, message, type, is_active) VALUES (?, ?, ?, ?)");
        $stmt->bind_param("sssi", $title, $message, $type, $isActive);
        if ($stmt->execute()) {
            header('Location: notifications.php?success=1');
            exit;
        }
        $stmt->close();
    }
}

// Handle toggle active status
if (isset($_GET['toggle'])) {
    $id = (int)$_GET['toggle'];
    $stmt = $conn->prepare("UPDATE notifications SET is_active = NOT is_active WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $stmt->close();
    header('Location: notifications.php');
    exit;
}

// Handle delete
if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    $stmt = $conn->prepare("DELETE FROM notifications WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $stmt->close();
    header('Location: notifications.php?success=1');
    exit;
}

// Get all notifications
$notifications = $conn->query("SELECT * FROM notifications ORDER BY created_at DESC")->fetch_all(MYSQLI_ASSOC);

$conn->close();

$pageTitle = 'Manage Notifications';
include 'layout_start.php';
?>

<div class="bg-white rounded-lg shadow-md p-4 md:p-6">
    <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4 mb-6">
        <h1 class="text-2xl md:text-3xl font-bold text-gray-800">
            <i class="fas fa-bell mr-2"></i>Manage Notifications
        </h1>
        <button onclick="document.getElementById('createModal').classList.remove('hidden')" 
                class="w-full sm:w-auto bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg flex items-center justify-center">
            <i class="fas fa-plus mr-2"></i>Create Notification
        </button>
    </div>

            <?php if (isset($_GET['success'])): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                    Operation successful!
                </div>
            <?php endif; ?>

            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Title</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Message</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <?php if (empty($notifications)): ?>
                            <tr>
                                <td colspan="6" class="px-6 py-4 text-center text-gray-500">No notifications found</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($notifications as $notif): ?>
                                <tr>
                                    <td class="px-6 py-4 whitespace-nowrap font-medium"><?= htmlspecialchars($notif['title']) ?></td>
                                    <td class="px-6 py-4"><?= htmlspecialchars(substr($notif['message'], 0, 50)) ?><?= strlen($notif['message']) > 50 ? '...' : '' ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="px-2 py-1 rounded text-xs font-medium 
                                            <?php 
                                            echo match($notif['type']) {
                                                'success' => 'bg-green-100 text-green-800',
                                                'warning' => 'bg-yellow-100 text-yellow-800',
                                                'error' => 'bg-red-100 text-red-800',
                                                default => 'bg-blue-100 text-blue-800'
                                            };
                                            ?>">
                                            <?= ucfirst($notif['type']) ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <a href="?toggle=<?= $notif['id'] ?>" 
                                           class="px-2 py-1 rounded text-xs font-medium <?= $notif['is_active'] ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800' ?>">
                                            <?= $notif['is_active'] ? 'Active' : 'Inactive' ?>
                                        </a>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <?= date('M d, Y', strtotime($notif['created_at'])) ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm">
                                        <a href="?delete=<?= $notif['id'] ?>" 
                                           onclick="return confirm('Are you sure you want to delete this notification?')"
                                           class="text-red-600 hover:text-red-900">
                                            <i class="fas fa-trash"></i> Delete
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Create Modal -->
    <div id="createModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-bold text-gray-900">Create Notification</h3>
                <button onclick="document.getElementById('createModal').classList.add('hidden')" 
                        class="text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <form method="POST" action="">
                <input type="hidden" name="action" value="create">
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Title</label>
                    <input type="text" name="title" required 
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                </div>
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Message</label>
                    <textarea name="message" required rows="4" 
                              class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"></textarea>
                </div>
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Type</label>
                    <select name="type" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        <option value="info">Info</option>
                        <option value="success">Success</option>
                        <option value="warning">Warning</option>
                        <option value="error">Error</option>
                    </select>
                </div>
                <div class="mb-4">
                    <label class="flex items-center">
                        <input type="checkbox" name="is_active" checked class="mr-2">
                        <span class="text-sm text-gray-700">Active</span>
                    </label>
                </div>
                <div class="flex justify-end gap-2">
                    <button type="button" 
                            onclick="document.getElementById('createModal').classList.add('hidden')" 
                            class="px-4 py-2 bg-gray-300 text-gray-700 rounded-md hover:bg-gray-400">
                        Cancel
                    </button>
                    <button type="submit" 
                            class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                        Create
                    </button>
                </div>
            </form>
        </div>
    </div>
<?php include 'layout_end.php'; ?>

