<?php
/**
 * OneSignal Notification Service
 * 
 * This file handles sending push notifications via OneSignal API
 * 
 * Required OneSignal credentials:
 * - APP_ID: Your OneSignal App ID
 * - REST_API_KEY: Your OneSignal REST API Key
 * 
 * Get these from: OneSignal Dashboard > Settings > Keys & IDs
 */

// Include config to access BASE_URL constant
if (!defined('BASE_URL')) {
    require_once __DIR__ . '/config.php';
}

class OneSignalService {
    private $appId;
    private $restApiKey;
    private $apiUrl = 'https://onesignal.com/api/v1/notifications';
    
    public function __construct($appId, $restApiKey) {
        $this->appId = $appId;
        $this->restApiKey = $restApiKey;
    }
    
    /**
     * Send notification to all users
     */
    public function sendToAll($title, $message, $data = [], $imageUrl = null, $launchUrl = null) {
        $fields = [
            'app_id' => $this->appId,
            'included_segments' => ['All'],
            'headings' => ['en' => $title],
            'contents' => ['en' => $message]
        ];
        
        // Add image URL if provided (support all platforms)
        if (!empty($imageUrl)) {
            // Ensure URL is absolute
            if (!preg_match('/^https?:\/\//', $imageUrl)) {
                // If relative URL, make it absolute based on BASE_URL
                $imageUrl = rtrim(BASE_URL, '/') . '/' . ltrim($imageUrl, '/');
            }
            
            $fields['big_picture'] = $imageUrl; // Android
            $fields['ios_attachments'] = ['id1' => $imageUrl]; // iOS (hash/object format)
            $fields['chrome_web_image'] = $imageUrl; // Chrome Web
            $fields['adm_big_picture'] = $imageUrl; // Amazon Fire OS
        }
        
        // Add launch URL if provided
        if (!empty($launchUrl)) {
            $fields['url'] = $launchUrl;
        }
        
        // Only add data field if it's a non-empty array/object
        if (!empty($data) && is_array($data)) {
            $fields['data'] = $data;
        }
        
        return $this->sendNotification($fields);
    }
    
    /**
     * Send notification to specific user by player_id
     */
    public function sendToPlayer($playerId, $title, $message, $data = [], $imageUrl = null, $launchUrl = null) {
        $fields = [
            'app_id' => $this->appId,
            'include_player_ids' => [$playerId],
            'headings' => ['en' => $title],
            'contents' => ['en' => $message]
        ];
        
        // Add image URL if provided (support all platforms)
        if (!empty($imageUrl)) {
            // Ensure URL is absolute
            if (!preg_match('/^https?:\/\//', $imageUrl)) {
                // If relative URL, make it absolute based on BASE_URL
                $imageUrl = rtrim(BASE_URL, '/') . '/' . ltrim($imageUrl, '/');
            }
            
            $fields['big_picture'] = $imageUrl; // Android
            $fields['ios_attachments'] = ['id1' => $imageUrl]; // iOS (hash/object format)
            $fields['chrome_web_image'] = $imageUrl; // Chrome Web
            $fields['adm_big_picture'] = $imageUrl; // Amazon Fire OS
        }
        
        // Add launch URL if provided
        if (!empty($launchUrl)) {
            $fields['url'] = $launchUrl;
        }
        
        // Only add data field if it's a non-empty array/object
        if (!empty($data) && is_array($data)) {
            $fields['data'] = $data;
        }
        
        return $this->sendNotification($fields);
    }
    
    /**
     * Send notification to specific user by email (using tags)
     */
    public function sendToEmail($email, $title, $message, $data = [], $imageUrl = null, $launchUrl = null) {
        $fields = [
            'app_id' => $this->appId,
            'filters' => [
                ['field' => 'tag', 'key' => 'email', 'relation' => '=', 'value' => $email]
            ],
            'headings' => ['en' => $title],
            'contents' => ['en' => $message]
        ];
        
        // Add image URL if provided (support all platforms)
        if (!empty($imageUrl)) {
            // Ensure URL is absolute
            if (!preg_match('/^https?:\/\//', $imageUrl)) {
                // If relative URL, make it absolute based on BASE_URL
                $imageUrl = rtrim(BASE_URL, '/') . '/' . ltrim($imageUrl, '/');
            }
            
            $fields['big_picture'] = $imageUrl; // Android
            $fields['ios_attachments'] = ['id1' => $imageUrl]; // iOS (hash/object format)
            $fields['chrome_web_image'] = $imageUrl; // Chrome Web
            $fields['adm_big_picture'] = $imageUrl; // Amazon Fire OS
        }
        
        // Add launch URL if provided
        if (!empty($launchUrl)) {
            $fields['url'] = $launchUrl;
        }
        
        // Only add data field if it's a non-empty array/object
        if (!empty($data) && is_array($data)) {
            $fields['data'] = $data;
        }
        
        return $this->sendNotification($fields);
    }
    
    /**
     * Send notification to multiple players
     */
    public function sendToPlayers($playerIds, $title, $message, $data = [], $imageUrl = null, $launchUrl = null) {
        $fields = [
            'app_id' => $this->appId,
            'include_player_ids' => $playerIds,
            'headings' => ['en' => $title],
            'contents' => ['en' => $message]
        ];
        
        // Add image URL if provided (support all platforms)
        if (!empty($imageUrl)) {
            // Ensure URL is absolute
            if (!preg_match('/^https?:\/\//', $imageUrl)) {
                // If relative URL, make it absolute based on BASE_URL
                $imageUrl = rtrim(BASE_URL, '/') . '/' . ltrim($imageUrl, '/');
            }
            
            $fields['big_picture'] = $imageUrl; // Android
            $fields['ios_attachments'] = ['id1' => $imageUrl]; // iOS (hash/object format)
            $fields['chrome_web_image'] = $imageUrl; // Chrome Web
            $fields['adm_big_picture'] = $imageUrl; // Amazon Fire OS
        }
        
        // Add launch URL if provided
        if (!empty($launchUrl)) {
            $fields['url'] = $launchUrl;
        }
        
        // Only add data field if it's a non-empty array/object
        if (!empty($data) && is_array($data)) {
            $fields['data'] = $data;
        }
        
        return $this->sendNotification($fields);
    }
    
    /**
     * Make API request to OneSignal
     */
    private function sendNotification($fields) {
        // Debug: Log the payload being sent (especially image URLs)
        if (isset($fields['big_picture'])) {
            error_log("OneSignal Payload - Image URL: " . $fields['big_picture']);
        }
        error_log("OneSignal Payload: " . json_encode($fields));
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $this->apiUrl);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json; charset=utf-8',
            'Authorization: Basic ' . $this->restApiKey
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HEADER, false);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($fields));
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200) {
            return ['success' => true, 'response' => json_decode($response, true)];
        } else {
            return ['success' => false, 'error' => $response, 'http_code' => $httpCode];
        }
    }
}

