<?php
require_once 'config.php';
startSession();

if (!isAdminLoggedIn()) {
    header('Location: login.php');
    exit;
}

$conn = getDbConnection();

// Handle mark as paid
if (isset($_GET['mark_paid'])) {
    $orderId = (int)$_GET['mark_paid'];
    
    $stmt = $conn->prepare("UPDATE orders SET status = 'paid' WHERE id = ?");
    $stmt->bind_param("i", $orderId);
    
    if ($stmt->execute()) {
        // Generate download token
        $token = bin2hex(random_bytes(32));
        $expiresAt = date('Y-m-d H:i:s', strtotime('+24 hours'));
        
        $tokenStmt = $conn->prepare("INSERT INTO download_tokens (order_id, token, expires_at) VALUES (?, ?, ?)");
        $tokenStmt->bind_param("iss", $orderId, $token, $expiresAt);
        $tokenStmt->execute();
        $tokenStmt->close();
        
        header('Location: orders.php?success=1');
        exit;
    }
    $stmt->close();
}

// Handle reject
if (isset($_GET['reject'])) {
    $orderId = (int)$_GET['reject'];
    $stmt = $conn->prepare("UPDATE orders SET status = 'rejected' WHERE id = ?");
    $stmt->bind_param("i", $orderId);
    $stmt->execute();
    $stmt->close();
    header('Location: orders.php?success=1');
    exit;
}

// Get orders
$orders = $conn->query("
    SELECT o.*, p.title as product_title, p.preview_image, u.name as user_name, u.email 
    FROM orders o 
    LEFT JOIN products p ON o.product_id = p.id 
    LEFT JOIN users u ON o.user_id = u.id 
    ORDER BY o.created_at DESC
")->fetch_all(MYSQLI_ASSOC);

$conn->close();

$pageTitle = 'Orders Management';
include 'layout_start.php';
?>

<div class="bg-white rounded-lg shadow-lg overflow-hidden mb-6">
    <div class="overflow-x-auto">
        <table class="w-full min-w-[800px]">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Order ID</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Product</th>
                            <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden md:table-cell">User</th>
                            <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Amount</th>
                            <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden lg:table-cell">Payment Method</th>
                            <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden lg:table-cell">Transaction ID</th>
                            <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                            <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden md:table-cell">Date</th>
                            <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-200">
                        <?php foreach ($orders as $order): ?>
                        <tr class="hover:bg-gray-50">
                            <td class="px-3 md:px-6 py-4 whitespace-nowrap text-sm font-medium">#<?php echo $order['id']; ?></td>
                            <td class="px-3 md:px-6 py-4 text-sm max-w-[150px] truncate"><?php echo htmlspecialchars($order['product_title'] ?? 'N/A'); ?></td>
                            <td class="px-3 md:px-6 py-4 text-sm hidden md:table-cell max-w-[120px] truncate"><?php echo htmlspecialchars($order['user_name'] ?? $order['email'] ?? 'Guest'); ?></td>
                            <td class="px-3 md:px-6 py-4 whitespace-nowrap text-sm">৳<?php echo number_format($order['amount'] ?? 0, 2); ?></td>
                            <td class="px-3 md:px-6 py-4 text-sm hidden lg:table-cell"><?php echo htmlspecialchars($order['payment_method'] ?? '-'); ?></td>
                            <td class="px-3 md:px-6 py-4 text-sm text-gray-600 hidden lg:table-cell max-w-[100px] truncate"><?php echo htmlspecialchars($order['payment_transaction_id'] ?? '-'); ?></td>
                            <td class="px-3 md:px-6 py-4 whitespace-nowrap">
                                <?php
                                $statusColors = [
                                    'pending' => 'bg-yellow-100 text-yellow-800',
                                    'paid' => 'bg-green-100 text-green-800',
                                    'rejected' => 'bg-red-100 text-red-800'
                                ];
                                $color = $statusColors[$order['status']] ?? 'bg-gray-100 text-gray-800';
                                ?>
                                <span class="px-2 py-1 rounded-full text-xs font-semibold <?php echo $color; ?>">
                                    <?php echo ucfirst($order['status']); ?>
                                </span>
                            </td>
                            <td class="px-3 md:px-6 py-4 whitespace-nowrap text-sm text-gray-500 hidden md:table-cell">
                                <?php echo date('M d, Y', strtotime($order['created_at'])); ?>
                            </td>
                            <td class="px-3 md:px-6 py-4 whitespace-nowrap">
                                <?php if ($order['status'] === 'pending'): ?>
                                    <div class="flex flex-col sm:flex-row gap-2">
                                        <a href="?mark_paid=<?php echo $order['id']; ?>" 
                                           class="text-green-600 hover:text-green-800 text-xs sm:text-sm"
                                           onclick="return confirm('Mark this order as paid?')">
                                            <i class="fas fa-check"></i> <span class="hidden sm:inline">Approve</span>
                                        </a>
                                        <a href="?reject=<?php echo $order['id']; ?>" 
                                           class="text-red-600 hover:text-red-800 text-xs sm:text-sm"
                                           onclick="return confirm('Reject this order?')">
                                            <i class="fas fa-times"></i> <span class="hidden sm:inline">Reject</span>
                                        </a>
                                    </div>
                                <?php else: ?>
                                    <span class="text-gray-400 text-xs">-</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

<?php include 'layout_end.php'; ?>

