<?php
require_once 'config.php';
startSession();

if (!isAdminLoggedIn()) {
    header('Location: login.php');
    exit;
}

$conn = getDbConnection();
$message = '';
$messageType = '';

if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    $stmt = $conn->prepare("DELETE FROM payment_methods WHERE id = ?");
    $stmt->bind_param("i", $id);
    if ($stmt->execute()) {
        $message = 'Payment method deleted successfully';
        $messageType = 'success';
    }
    $stmt->close();
}

// Handle save
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_payment'])) {
    $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
    $name = sanitize($_POST['name'] ?? '');
    $number = sanitize($_POST['number'] ?? '');
    $instructions = sanitize($_POST['instructions'] ?? '');
    $isActive = isset($_POST['is_active']) ? 1 : 0;

    if (empty($name) || empty($number)) {
        $message = 'Name and number are required';
        $messageType = 'error';
    } else {
        if ($id > 0) {
            $stmt = $conn->prepare("UPDATE payment_methods SET name = ?, number = ?, instructions = ?, is_active = ? WHERE id = ?");
            $stmt->bind_param("sssii", $name, $number, $instructions, $isActive, $id);
        } else {
            $stmt = $conn->prepare("INSERT INTO payment_methods (name, number, instructions, is_active) VALUES (?, ?, ?, ?)");
            $stmt->bind_param("sssi", $name, $number, $instructions, $isActive);
        }

        if ($stmt->execute()) {
            $message = 'Payment method saved successfully';
            $messageType = 'success';
            header('Location: payment_methods.php');
            exit;
        } else {
            $message = 'Failed to save payment method';
            $messageType = 'error';
        }
        $stmt->close();
    }
}

$paymentMethods = $conn->query("SELECT * FROM payment_methods ORDER BY name")->fetch_all(MYSQLI_ASSOC);
$conn->close();

$pageTitle = 'Payment Methods';
include 'layout_start.php';
?>

<div class="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4 mb-6">
    <h2 class="text-2xl md:text-3xl font-bold text-gray-800">Payment Methods</h2>
    <button onclick="openModal()" class="w-full sm:w-auto bg-teal-600 hover:bg-teal-700 text-white px-4 md:px-6 py-2 md:py-3 rounded-lg flex items-center justify-center">
        <i class="fas fa-plus mr-2"></i> Add Payment Method
    </button>
</div>

        <?php if ($message): ?>
            <div class="mb-4 p-4 rounded-lg <?php echo $messageType === 'success' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <div class="bg-white rounded-lg shadow-lg overflow-hidden">
            <table class="w-full">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">ID</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Name</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Number</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-gray-200">
                    <?php foreach ($paymentMethods as $method): ?>
                    <tr>
                        <td class="px-6 py-4">#<?php echo $method['id']; ?></td>
                        <td class="px-6 py-4 font-medium"><?php echo htmlspecialchars($method['name']); ?></td>
                        <td class="px-6 py-4"><?php echo htmlspecialchars($method['number']); ?></td>
                        <td class="px-6 py-4">
                            <span class="px-2 py-1 rounded-full text-xs font-semibold <?php echo $method['is_active'] ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                                <?php echo $method['is_active'] ? 'Active' : 'Inactive'; ?>
                            </span>
                        </td>
                        <td class="px-6 py-4">
                            <button onclick="editMethod(<?php echo htmlspecialchars(json_encode($method)); ?>)" 
                                    class="text-blue-600 hover:text-blue-800 mr-3">
                                <i class="fas fa-edit"></i>
                            </button>
                            <a href="?delete=<?php echo $method['id']; ?>" 
                               onclick="return confirm('Are you sure?')"
                               class="text-red-600 hover:text-red-800">
                                <i class="fas fa-trash"></i>
                            </a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Modal -->
    <div id="methodModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
        <div class="bg-white rounded-lg p-8 max-w-md w-full mx-4">
            <div class="flex justify-between items-center mb-6">
                <h3 class="text-2xl font-bold" id="modalTitle">Add Payment Method</h3>
                <button onclick="closeModal()" class="text-gray-500 hover:text-gray-700">
                    <i class="fas fa-times text-2xl"></i>
                </button>
            </div>

            <form method="POST" action="">
                <input type="hidden" name="id" id="methodId">
                <input type="hidden" name="save_payment" value="1">
                
                <div class="mb-4">
                    <label class="block text-gray-700 font-medium mb-2">Payment Method Name</label>
                    <input type="text" name="name" id="methodName" required 
                           class="w-full px-4 py-2 border rounded-lg" placeholder="e.g., bKash, Nagad">
                </div>

                <div class="mb-4">
                    <label class="block text-gray-700 font-medium mb-2">Account Number</label>
                    <input type="text" name="number" id="methodNumber" required 
                           class="w-full px-4 py-2 border rounded-lg">
                </div>

                <div class="mb-4">
                    <label class="block text-gray-700 font-medium mb-2">Instructions</label>
                    <textarea name="instructions" id="methodInstructions" rows="3" 
                              class="w-full px-4 py-2 border rounded-lg" 
                              placeholder="Payment instructions for users"></textarea>
                </div>

                <div class="mb-4">
                    <label class="flex items-center">
                        <input type="checkbox" name="is_active" id="methodActive" value="1" checked class="mr-2">
                        <span class="text-gray-700">Active</span>
                    </label>
                </div>

                <div class="flex justify-end space-x-4">
                    <button type="button" onclick="closeModal()" 
                            class="px-6 py-2 border rounded-lg hover:bg-gray-50">
                        Cancel
                    </button>
                    <button type="submit" 
                            class="px-6 py-2 bg-teal-600 text-white rounded-lg hover:bg-teal-700">
                        Save
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function openModal(method = null) {
            document.getElementById('methodModal').classList.remove('hidden');
            if (method) {
                document.getElementById('modalTitle').textContent = 'Edit Payment Method';
                document.getElementById('methodId').value = method.id;
                document.getElementById('methodName').value = method.name || '';
                document.getElementById('methodNumber').value = method.number || '';
                document.getElementById('methodInstructions').value = method.instructions || '';
                document.getElementById('methodActive').checked = method.is_active == 1;
            } else {
                document.getElementById('modalTitle').textContent = 'Add Payment Method';
                document.getElementById('methodForm').reset();
            }
        }

        function closeModal() {
            document.getElementById('methodModal').classList.add('hidden');
        }

        function editMethod(method) {
            openModal(method);
        }
    </script>

<?php include 'layout_end.php'; ?>

