<?php
require_once 'config.php';
startSession();

if (!isAdminLoggedIn()) {
    header('Location: login.php');
    exit;
}

$conn = getDbConnection();
$message = '';
$messageType = '';

// Handle delete
if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    $stmt = $conn->prepare("DELETE FROM products WHERE id = ?");
    $stmt->bind_param("i", $id);
    if ($stmt->execute()) {
        $message = 'Product deleted successfully';
        $messageType = 'success';
    } else {
        $message = 'Failed to delete product';
        $messageType = 'error';
    }
    $stmt->close();
}

// Get all products
$products = $conn->query("
    SELECT p.*, c.name as category_name 
    FROM products p 
    LEFT JOIN categories c ON p.category_id = c.id 
    ORDER BY p.created_at DESC
")->fetch_all(MYSQLI_ASSOC);

// Get categories for form
$categories = $conn->query("SELECT * FROM categories ORDER BY name")->fetch_all(MYSQLI_ASSOC);

$conn->close();

$pageTitle = 'Products Management';
include 'layout_start.php';
?>

<div class="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4 mb-6">
    <h2 class="text-2xl md:text-3xl font-bold text-gray-800">Products Management</h2>
    <button onclick="openModal()" class="w-full sm:w-auto bg-teal-600 hover:bg-teal-700 text-white px-4 md:px-6 py-2 md:py-3 rounded-lg flex items-center justify-center">
        <i class="fas fa-plus mr-2"></i> Add New Product
    </button>
</div>

        <?php if ($message): ?>
            <div class="mb-4 p-4 rounded-lg <?php echo $messageType === 'success' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

<div class="bg-white rounded-lg shadow-lg overflow-hidden">
    <div class="overflow-x-auto">
        <table class="w-full min-w-[800px]">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">ID</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Image</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Title</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden md:table-cell">Category</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Price</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden sm:table-cell">Status</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($products as $product): ?>
                <tr class="hover:bg-gray-50">
                    <td class="px-3 md:px-6 py-4 whitespace-nowrap text-sm font-medium">#<?php echo $product['id']; ?></td>
                    <td class="px-3 md:px-6 py-4">
                        <?php if ($product['preview_image']): ?>
                            <img src="api/uploads/<?php echo htmlspecialchars($product['preview_image']); ?>" 
                                 alt="" class="w-12 h-12 md:w-16 md:h-16 object-cover rounded">
                        <?php else: ?>
                            <div class="w-12 h-12 md:w-16 md:h-16 bg-gray-200 rounded flex items-center justify-center">
                                <i class="fas fa-image text-gray-400 text-xs md:text-base"></i>
                            </div>
                        <?php endif; ?>
                    </td>
                    <td class="px-3 md:px-6 py-4 text-sm font-medium max-w-[150px] truncate"><?php echo htmlspecialchars($product['title']); ?></td>
                    <td class="px-3 md:px-6 py-4 text-sm hidden md:table-cell"><?php echo htmlspecialchars($product['category_name'] ?? 'Uncategorized'); ?></td>
                    <td class="px-3 md:px-6 py-4 whitespace-nowrap text-sm">
                        <?php if ($product['is_free']): ?>
                            <span class="text-blue-600 font-semibold">FREE</span>
                        <?php else: ?>
                            ৳<?php echo number_format($product['price'], 2); ?>
                        <?php endif; ?>
                    </td>
                    <td class="px-3 md:px-6 py-4 whitespace-nowrap hidden sm:table-cell">
                        <span class="px-2 py-1 rounded-full text-xs font-semibold bg-green-100 text-green-800">
                            Active
                        </span>
                    </td>
                    <td class="px-3 md:px-6 py-4 whitespace-nowrap">
                        <div class="flex items-center space-x-2">
                            <button onclick="editProduct(<?php echo htmlspecialchars(json_encode($product)); ?>)" 
                                    class="text-blue-600 hover:text-blue-800 p-1">
                                <i class="fas fa-edit text-sm md:text-base"></i>
                            </button>
                            <a href="?delete=<?php echo $product['id']; ?>" 
                               onclick="return confirm('Are you sure?')"
                               class="text-red-600 hover:text-red-800 p-1">
                                <i class="fas fa-trash text-sm md:text-base"></i>
                            </a>
                        </div>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>
    </div>

    <!-- Modal -->
    <div id="productModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
        <div class="bg-white rounded-lg p-8 max-w-2xl w-full mx-4 max-h-screen overflow-y-auto">
            <div class="flex justify-between items-center mb-6">
                <h3 class="text-2xl font-bold" id="modalTitle">Add New Product</h3>
                <button onclick="closeModal()" class="text-gray-500 hover:text-gray-700">
                    <i class="fas fa-times text-2xl"></i>
                </button>
            </div>

            <form id="productForm" action="api/product_save.php" method="POST" enctype="multipart/form-data">
                <input type="hidden" name="id" id="productId">
                
                <!-- Basic Information Section -->
                <div class="mb-6">
                    <h4 class="text-lg font-semibold text-gray-800 mb-4 border-b pb-2">Basic Information</h4>
                    
                    <div class="mb-4">
                        <label class="block text-gray-700 font-medium mb-2">Product Title <span class="text-red-500">*</span></label>
                        <input type="text" name="title" id="title" required 
                               class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-teal-500 focus:border-teal-500"
                               placeholder="Enter product title">
                    </div>

                    <div class="grid grid-cols-2 gap-4 mb-4">
                        <div>
                            <label class="block text-gray-700 font-medium mb-2">Category <span class="text-red-500">*</span></label>
                            <select name="category_id" id="categoryId" required 
                                    class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-teal-500 focus:border-teal-500">
                                <option value="">Select Category</option>
                                <?php foreach ($categories as $cat): ?>
                                    <option value="<?php echo $cat['id']; ?>"><?php echo htmlspecialchars($cat['name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div>
                            <label class="block text-gray-700 font-medium mb-2">Product Type <span class="text-red-500">*</span></label>
                            <select name="product_type" id="productType" onchange="togglePriceField()" required 
                                    class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-teal-500 focus:border-teal-500">
                                <option value="paid">Paid</option>
                                <option value="free">Free</option>
                            </select>
                        </div>
                    </div>

                    <div id="priceField" class="mb-4">
                        <label class="block text-gray-700 font-medium mb-2">Price <span class="text-red-500">*</span></label>
                        <input type="number" step="0.01" min="0" name="price" id="price" 
                               class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-teal-500 focus:border-teal-500"
                               placeholder="0.00">
                        <p class="text-xs text-gray-500 mt-1">Enter the price for paid products</p>
                    </div>
                </div>

                <!-- Media Section -->
                <div class="mb-6">
                    <h4 class="text-lg font-semibold text-gray-800 mb-4 border-b pb-2">Media & Files</h4>
                    
                    <div class="mb-4">
                        <label class="block text-gray-700 font-medium mb-2">Preview Image</label>
                        <input type="file" name="preview_image_upload" id="previewImageUpload" accept="image/jpeg,image/png,image/gif,image/webp,image/jpg"
                               class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-teal-500 focus:border-teal-500">
                        <p class="text-xs text-gray-500 mt-1">Upload image file (JPEG, PNG, GIF, WebP - Max 5MB)</p>
                        <input type="hidden" name="preview_image" id="previewImage">
                        <!-- Image Preview -->
                        <div id="imagePreview" class="mt-3 hidden">
                            <img id="previewImageDisplay" src="" alt="Preview" 
                                 class="max-w-full h-48 object-contain border rounded-lg">
                        </div>
                        <!-- Existing Image Display (for edit mode) -->
                        <div id="existingImagePreview" class="mt-3 hidden">
                            <p class="text-xs text-gray-600 mb-2">Current Image:</p>
                            <img id="existingImageDisplay" src="" alt="Current" 
                                 class="max-w-full h-48 object-contain border rounded-lg">
                        </div>
                    </div>

                    <div class="mb-4">
                        <label class="block text-gray-700 font-medium mb-2">Download URL</label>
                        <input type="text" name="download_url" id="downloadUrl" 
                               class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-teal-500 focus:border-teal-500" 
                               placeholder="uploads/file.zip or https://example.com/file.zip">
                        <p class="text-xs text-gray-500 mt-1">URL or path to the downloadable file</p>
                    </div>
                </div>

                <!-- Description Section -->
                <div class="mb-6">
                    <h4 class="text-lg font-semibold text-gray-800 mb-4 border-b pb-2">Description</h4>
                    
                    <div class="mb-4">
                        <label class="block text-gray-700 font-medium mb-2">Product Description</label>
                        <textarea name="description" id="description" rows="5" 
                                  class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-teal-500 focus:border-teal-500"
                                  placeholder="Enter detailed product description"></textarea>
                        <p class="text-xs text-gray-500 mt-1">Describe your product in detail</p>
                    </div>
                </div>

                <div class="flex justify-end space-x-4">
                    <button type="button" onclick="closeModal()" 
                            class="px-6 py-2 border rounded-lg hover:bg-gray-50">
                        Cancel
                    </button>
                    <button type="submit" 
                            class="px-6 py-2 bg-teal-600 text-white rounded-lg hover:bg-teal-700">
                        Save Product
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function togglePriceField() {
            const productType = document.getElementById('productType').value;
            const priceField = document.getElementById('priceField');
            const priceInput = document.getElementById('price');
            
            if (productType === 'free') {
                priceField.style.display = 'none';
                priceInput.removeAttribute('required');
                priceInput.value = '0';
            } else {
                priceField.style.display = 'block';
                priceInput.setAttribute('required', 'required');
            }
        }
        
        function openModal(product = null) {
            document.getElementById('productModal').classList.remove('hidden');
            if (product) {
                document.getElementById('modalTitle').textContent = 'Edit Product';
                document.getElementById('productId').value = product.id;
                document.getElementById('categoryId').value = product.category_id || '';
                document.getElementById('title').value = product.title || '';
                document.getElementById('description').value = product.description || '';
                document.getElementById('price').value = product.price || '';
                document.getElementById('previewImage').value = product.preview_image || '';
                document.getElementById('downloadUrl').value = product.download_url || '';
                
                // Handle existing preview image
                if (product.preview_image) {
                    const existingDiv = document.getElementById('existingImagePreview');
                    const existingImg = document.getElementById('existingImageDisplay');
                    let imgUrl = product.preview_image;
                    // Convert relative path to full URL if needed
                    if (!imgUrl.startsWith('http')) {
                        imgUrl = 'api/uploads/' + imgUrl;
                    }
                    existingImg.src = imgUrl;
                    existingDiv.classList.remove('hidden');
                } else {
                    document.getElementById('existingImagePreview').classList.add('hidden');
                }
                // Set product type based on is_free
                document.getElementById('productType').value = product.is_free == 1 ? 'free' : 'paid';
                togglePriceField(); // Update price field visibility
            } else {
                document.getElementById('modalTitle').textContent = 'Add New Product';
                document.getElementById('productForm').reset();
                document.getElementById('productType').value = 'paid'; // Default to paid
                togglePriceField(); // Update price field visibility
            }
        }

        function closeModal() {
            document.getElementById('productModal').classList.add('hidden');
        }

        function editProduct(product) {
            openModal(product);
        }
        
        // Image preview functionality
        function setupImagePreview() {
            const uploadInput = document.getElementById('previewImageUpload');
            const previewDiv = document.getElementById('imagePreview');
            const previewImg = document.getElementById('previewImageDisplay');
            const existingDiv = document.getElementById('existingImagePreview');
            
            if (uploadInput) {
                uploadInput.addEventListener('change', function(e) {
                    const file = e.target.files[0];
                    if (file) {
                        // Validate file type
                        const allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/jpg'];
                        if (!allowedTypes.includes(file.type)) {
                            alert('Invalid image type. Allowed: JPEG, PNG, GIF, WebP');
                            e.target.value = '';
                            return;
                        }
                        
                        // Validate file size (5MB)
                        if (file.size > 5 * 1024 * 1024) {
                            alert('Image size too large. Maximum 5MB allowed.');
                            e.target.value = '';
                            return;
                        }
                        
                        // Hide existing image preview
                        if (existingDiv) {
                            existingDiv.classList.add('hidden');
                        }
                        
                        // Show new image preview
                        const reader = new FileReader();
                        reader.onload = function(event) {
                            previewImg.src = event.target.result;
                            previewDiv.classList.remove('hidden');
                        };
                        reader.readAsDataURL(file);
                    } else {
                        previewDiv.classList.add('hidden');
                    }
                });
            }
        }
        
        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            togglePriceField();
            setupImagePreview();
        });
    </script>
</body>
</html>

