<?php
require_once 'config.php';
require_once 'onesignal_service.php';
startSession();

if (!isAdminLoggedIn()) {
    header('Location: login.php');
    exit;
}

$message = '';
$messageType = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $title = sanitize($_POST['title'] ?? '');
    $messageText = sanitize($_POST['message'] ?? '');
    $sendTo = $_POST['send_to'] ?? 'all';
    $targetEmail = sanitize($_POST['target_email'] ?? '');
    $launchUrl = sanitize($_POST['launch_url'] ?? '');
    
    // Handle image upload
    $uploadedImageUrl = null;
    $uploadError = false;
    
    if (isset($_FILES['image_upload']) && $_FILES['image_upload']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = UPLOAD_DIR;
        // Create upload directory if it doesn't exist
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        $file = $_FILES['image_upload'];
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/jpg'];
        $maxSize = 5 * 1024 * 1024; // 5MB
        
        // Validate file type
        if (!in_array($file['type'], $allowedTypes)) {
            $message = 'Invalid image type. Allowed: JPEG, PNG, GIF, WebP';
            $messageType = 'error';
            $uploadError = true;
        } elseif ($file['size'] > $maxSize) {
            $message = 'Image size too large. Maximum 5MB allowed.';
            $messageType = 'error';
            $uploadError = true;
        } else {
            // Generate unique filename
            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = 'notification_' . time() . '_' . uniqid() . '.' . $extension;
            $filePath = $uploadDir . $filename;
            
            // Move uploaded file
            if (move_uploaded_file($file['tmp_name'], $filePath)) {
                // Generate full absolute URL for OneSignal
                // Get current domain and protocol dynamically
                $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
                $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
                $scriptDir = dirname($_SERVER['PHP_SELF']); // e.g., /admin or /admin/
                $scriptDir = rtrim($scriptDir, '/'); // Remove trailing slash
                
                // Build absolute URL
                $uploadedImageUrl = $protocol . '://' . $host . $scriptDir . '/img/push_notifications/' . $filename;
                
                error_log("Generated Image URL: " . $uploadedImageUrl);
            } else {
                $message = 'Failed to upload image.';
                $messageType = 'error';
                $uploadError = true;
            }
        }
    }
    
    if (empty($title) || empty($messageText)) {
        $message = 'Title and message are required!';
        $messageType = 'error';
    } elseif (!$uploadError) { // Only proceed if no error occurred during upload
        // Initialize OneSignal service
        $oneSignal = new OneSignalService(ONESIGNAL_APP_ID, ONESIGNAL_REST_API_KEY);
        
        // Prepare image and launch URLs - use uploaded image if available
        $imgUrl = !empty($uploadedImageUrl) ? $uploadedImageUrl : null;
        $launch = !empty($launchUrl) ? $launchUrl : null;
        
        // Debug: Show image URL in response for verification
        if (!empty($imgUrl)) {
            error_log("OneSignal Image URL: " . $imgUrl);
            // Verify image URL is accessible
            $headers = @get_headers($imgUrl);
            if ($headers && strpos($headers[0], '200') !== false) {
                error_log("Image URL is accessible: " . $imgUrl);
            } else {
                error_log("Image URL may not be accessible: " . $imgUrl);
            }
        }
        
        if ($sendTo === 'all') {
            // Send to all users
            $result = $oneSignal->sendToAll($title, $messageText, [], $imgUrl, $launch);
            if ($result['success']) {
                $message = 'Notification sent to all users successfully!';
                if (!empty($imgUrl)) {
                    $message .= ' Image URL: ' . htmlspecialchars($imgUrl);
                }
                $messageType = 'success';
            } else {
                $message = 'Failed to send notification: ' . ($result['error'] ?? 'Unknown error');
                $messageType = 'error';
            }
        } elseif ($sendTo === 'email') {
            // Send to specific email
            if (empty($targetEmail)) {
                $message = 'Email is required!';
                $messageType = 'error';
            } else {
                $result = $oneSignal->sendToEmail($targetEmail, $title, $messageText, [], $imgUrl, $launch);
                if ($result['success']) {
                    $message = 'Notification sent to email: ' . htmlspecialchars($targetEmail) . ' successfully!';
                    if (!empty($imgUrl)) {
                        $message .= ' Image URL: ' . htmlspecialchars($imgUrl);
                    }
                    $messageType = 'success';
                } else {
                    $message = 'Failed to send notification: ' . ($result['error'] ?? 'Unknown error');
                    $messageType = 'error';
                }
            }
        }
    }
}

// Get all users for reference
$conn = getDbConnection();
$users = $conn->query("SELECT id, name, email FROM users ORDER BY created_at DESC LIMIT 50")->fetch_all(MYSQLI_ASSOC);
$conn->close();

$pageTitle = 'Push Notifications';
include 'layout_start.php';
?>

<h2 class="text-2xl md:text-3xl font-bold text-gray-800 mb-6">Push Notifications</h2>

        <?php if ($message): ?>
        <div class="mb-6 p-4 rounded-lg <?php echo $messageType === 'success' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
            <?php echo htmlspecialchars($message); ?>
        </div>
        <?php endif; ?>

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <!-- Send Notification Form -->
            <div class="lg:col-span-2">
                <div class="bg-white rounded-lg shadow-lg p-6">
                    <h3 class="text-xl font-bold text-gray-800 mb-4">Send Notification</h3>
                    
                    <form method="POST" id="notificationForm" enctype="multipart/form-data">
                        <div class="mb-4">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Send To</label>
                            <select name="send_to" id="sendTo" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" onchange="updateFormFields()">
                                <option value="all">All Users</option>
                                <option value="email">Specific Email</option>
                            </select>
                        </div>

                        <div class="mb-4" id="emailField" style="display: none;">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Target Email</label>
                            <input type="email" name="target_email" id="target_email" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" placeholder="user@example.com">
                            <p class="text-xs text-gray-500 mt-1">User's email address</p>
                        </div>

                        <div class="mb-4">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Title <span class="text-red-500">*</span></label>
                            <input type="text" name="title" required class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" placeholder="Notification title">
                        </div>

                        <div class="mb-4">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Message <span class="text-red-500">*</span></label>
                            <textarea name="message" required rows="5" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" placeholder="Notification message"></textarea>
                        </div>

                        <div class="mb-4">
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                <i class="fas fa-upload mr-2"></i>Upload Image (Optional)
                            </label>
                            <input type="file" name="image_upload" accept="image/jpeg,image/png,image/gif,image/webp" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                            <p class="text-xs text-gray-500 mt-1">Upload an image file (JPEG, PNG, GIF, WebP - Max 5MB)</p>
                        </div>

                        <div class="mb-4">
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                <i class="fas fa-link mr-2"></i>Launch URL (Optional)
                            </label>
                            <input type="url" name="launch_url" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" placeholder="https://example.com or app://deep-link">
                            <p class="text-xs text-gray-500 mt-1">URL to open when notification is clicked (web URL or deep link)</p>
                        </div>

                        <button type="submit" class="w-full bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition">
                            <i class="fas fa-paper-plane mr-2"></i> Send Notification
                        </button>
                    </form>
                </div>
            </div>

            <!-- Users Reference -->
            <div class="lg:col-span-1">
                <div class="bg-white rounded-lg shadow-lg p-6">
                    <h3 class="text-xl font-bold text-gray-800 mb-4">Recent Users</h3>
                    <div class="space-y-3 max-h-96 overflow-y-auto">
                        <?php if (empty($users)): ?>
                            <p class="text-gray-500 text-sm">No users with OneSignal data yet.</p>
                        <?php else: ?>
                            <?php foreach ($users as $user): ?>
                            <div class="p-3 border border-gray-200 rounded-lg hover:bg-gray-50 cursor-pointer" onclick="fillUserEmail('<?php echo htmlspecialchars($user['email']); ?>')">
                                <div class="font-medium text-gray-800"><?php echo htmlspecialchars($user['name']); ?></div>
                                <div class="text-xs text-gray-500"><?php echo htmlspecialchars($user['email']); ?></div>
                            </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        function updateFormFields() {
            const sendTo = document.getElementById('sendTo').value;
            const emailField = document.getElementById('emailField');
            
            emailField.style.display = sendTo === 'email' ? 'block' : 'none';
            
            // Clear field when hidden
            if (sendTo !== 'email') document.getElementById('target_email').value = '';
        }
        
        function fillUserEmail(email) {
            document.getElementById('target_email').value = email;
            // Auto-select email option
            document.getElementById('sendTo').value = 'email';
            updateFormFields();
        }
    </script>

<?php include 'layout_end.php'; ?>

