<?php
require_once 'config.php';
startSession();

if (!isAdminLoggedIn()) {
    header('Location: login.php');
    exit;
}

$conn = getDbConnection();
$message = '';
$messageType = '';

// Handle add/edit
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $version = sanitize($_POST['version'] ?? '');
    $updateMessage = sanitize($_POST['update_message'] ?? '');
    $forceUpdate = isset($_POST['force_update']) ? 1 : 0;
    $isActive = isset($_POST['is_active']) ? 1 : 0;
    
    if (empty($version)) {
        $message = 'Version is required';
        $messageType = 'error';
    } else {
        // Handle APK URL input (replaced file upload)
        $apkUrl = trim($_POST['apk_url'] ?? '');
        
        // Validate URL if provided
        if (!empty($apkUrl)) {
            // Basic URL validation
            if (!filter_var($apkUrl, FILTER_VALIDATE_URL)) {
                $message = 'Please enter a valid APK download URL';
                $messageType = 'error';
            } elseif (strpos($apkUrl, '.apk') === false && strpos(strtolower($apkUrl), 'apk') === false) {
                // Warn if URL doesn't seem to point to APK (non-blocking)
                // Just a warning, not an error
            }
        } else {
            // APK URL is optional
            $apkUrl = null;
        }
        
        if ($messageType !== 'error') {
            if (isset($_POST['id']) && !empty($_POST['id'])) {
                // Edit existing version
                $id = (int)$_POST['id'];
                
                // If new APK uploaded, update URL, otherwise keep old one
                if ($apkUrl) {
                    $stmt = $conn->prepare("UPDATE app_versions SET version = ?, update_message = ?, apk_download_url = ?, force_update = ?, is_active = ? WHERE id = ?");
                    $stmt->bind_param("sssiii", $version, $updateMessage, $apkUrl, $forceUpdate, $isActive, $id);
                } else {
                    $stmt = $conn->prepare("UPDATE app_versions SET version = ?, update_message = ?, force_update = ?, is_active = ? WHERE id = ?");
                    $stmt->bind_param("ssiii", $version, $updateMessage, $forceUpdate, $isActive, $id);
                }
                
                if ($stmt->execute()) {
                    $message = 'Version updated successfully';
                    $messageType = 'success';
                } else {
                    $message = 'Failed to update version';
                    $messageType = 'error';
                }
                $stmt->close();
            } else {
                // Add new version
                // Deactivate all other versions
                $conn->query("UPDATE app_versions SET is_active = 0");
                
                if ($apkUrl) {
                    $stmt = $conn->prepare("INSERT INTO app_versions (version, update_message, apk_download_url, force_update, is_active) VALUES (?, ?, ?, ?, ?)");
                    $stmt->bind_param("sssii", $version, $updateMessage, $apkUrl, $forceUpdate, $isActive);
                } else {
                    $stmt = $conn->prepare("INSERT INTO app_versions (version, update_message, force_update, is_active) VALUES (?, ?, ?, ?)");
                    $stmt->bind_param("ssii", $version, $updateMessage, $forceUpdate, $isActive);
                }
                
                if ($stmt->execute()) {
                    $message = 'Version added successfully';
                    $messageType = 'success';
                } else {
                    $message = 'Failed to add version';
                    $messageType = 'error';
                }
                $stmt->close();
            }
        }
    }
}

// Get version for editing
$editVersion = null;
if (isset($_GET['edit'])) {
    $id = (int)$_GET['edit'];
    $result = $conn->query("SELECT * FROM app_versions WHERE id = $id");
    if ($result->num_rows > 0) {
        $editVersion = $result->fetch_assoc();
    }
}

// Get all versions
$versions = $conn->query("SELECT * FROM app_versions ORDER BY release_date DESC")->fetch_all(MYSQLI_ASSOC);

// Get base URL
$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://" . $_SERVER['HTTP_HOST'];

$conn->close();

$pageTitle = 'App Version Management';
include 'layout_start.php';
?>

<div class="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4 mb-6">
    <h2 class="text-2xl md:text-3xl font-bold text-gray-800">App Version Management</h2>
    <button onclick="openAddModal()" class="w-full sm:w-auto bg-teal-600 text-white px-4 py-2 rounded-lg hover:bg-teal-700 flex items-center justify-center">
        <span class="material-icons mr-2" style="font-size: 18px;">add</span> Add New Version
    </button>
</div>

        <?php if ($message): ?>
            <div class="mb-4 p-4 rounded-lg <?php echo $messageType === 'success' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

<div class="bg-white rounded-lg shadow-lg overflow-hidden">
    <div class="overflow-x-auto">
        <table class="min-w-full divide-y divide-gray-200">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Version</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden md:table-cell">Update Message</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden lg:table-cell">APK URL</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden sm:table-cell">Force Update</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden lg:table-cell">Release Date</th>
                    <th class="px-3 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                </tr>
            </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php if (empty($versions)): ?>
                        <tr>
                            <td colspan="7" class="px-6 py-4 text-center text-gray-500">No versions found</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($versions as $ver): ?>
                            <tr class="<?php echo $ver['is_active'] ? 'bg-green-50' : ''; ?>">
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($ver['version']); ?></span>
                                    <?php if ($ver['is_active']): ?>
                                        <span class="ml-2 px-2 py-1 text-xs bg-green-100 text-green-800 rounded">Active</span>
                                    <?php endif; ?>
                                </td>
                                <td class="px-6 py-4">
                                    <span class="text-sm text-gray-900"><?php echo htmlspecialchars($ver['update_message'] ?? '-'); ?></span>
                                </td>
                                <td class="px-6 py-4">
                                    <?php if ($ver['apk_download_url']): ?>
                                        <a href="<?php echo htmlspecialchars($ver['apk_download_url']); ?>" target="_blank" 
                                           class="text-teal-600 hover:text-teal-900 text-sm">
                                            <span class="material-icons" style="font-size: 16px; vertical-align: middle;">download</span> Download
                                        </a>
                                    <?php else: ?>
                                        <span class="text-sm text-gray-400">No APK</span>
                                    <?php endif; ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 py-1 text-xs rounded <?php echo $ver['force_update'] ? 'bg-red-100 text-red-800' : 'bg-gray-100 text-gray-800'; ?>">
                                        <?php echo $ver['force_update'] ? 'Yes' : 'No'; ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 py-1 text-xs rounded <?php echo $ver['is_active'] ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800'; ?>">
                                        <?php echo $ver['is_active'] ? 'Active' : 'Inactive'; ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                    <?php echo date('M d, Y H:i', strtotime($ver['release_date'])); ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                    <a href="?edit=<?php echo $ver['id']; ?>" class="text-teal-600 hover:text-teal-900 mr-3 inline-flex items-center">
                                        <span class="material-icons" style="font-size: 18px; margin-right: 4px;">edit</span> Edit
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Add/Edit Modal -->
<div id="versionModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
    <div class="bg-white rounded-lg p-4 md:p-6 w-full max-w-2xl max-h-[90vh] overflow-y-auto">
        <div class="flex justify-between items-center mb-4">
            <h3 class="text-lg md:text-xl font-bold"><?php echo $editVersion ? 'Edit' : 'Add'; ?> Version</h3>
            <button onclick="closeModal()" class="text-gray-500 hover:text-gray-700 p-1">
                <span class="material-icons">close</span>
            </button>
        </div>
            
            <form method="POST" action="">
                <?php if ($editVersion): ?>
                    <input type="hidden" name="id" value="<?php echo $editVersion['id']; ?>">
                <?php endif; ?>
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Version <span class="text-red-500">*</span></label>
                    <input type="text" name="version" required
                           value="<?php echo htmlspecialchars($editVersion['version'] ?? ''); ?>"
                           placeholder="e.g., 1.0.5"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-teal-500">
                    <p class="text-xs text-gray-500 mt-1">Format: major.minor.patch (e.g., 1.0.5)</p>
                </div>

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Update Message</label>
                    <textarea name="update_message" rows="3"
                              placeholder="e.g., Bug fixes and performance improvements."
                              class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-teal-500"><?php echo htmlspecialchars($editVersion['update_message'] ?? ''); ?></textarea>
                </div>

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        APK Download URL
                    </label>
                    <input type="url" name="apk_url"
                           value="<?php echo htmlspecialchars($editVersion['apk_download_url'] ?? ''); ?>"
                           placeholder="https://example.com/app/app-release.apk"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-teal-500">
                    <p class="text-xs text-gray-500 mt-1">
                        Enter the direct download URL to your APK file. You can use any hosting service (Google Drive, Dropbox, your own server, etc.)
                    </p>
                    <?php if ($editVersion && $editVersion['apk_download_url']): ?>
                        <p class="text-xs text-teal-600 mt-2">
                            <span class="material-icons" style="font-size: 14px; vertical-align: middle;">link</span>
                            <a href="<?php echo htmlspecialchars($editVersion['apk_download_url']); ?>" target="_blank" class="underline">Test Current URL</a>
                        </p>
                    <?php endif; ?>
                </div>

                <div class="mb-4">
                    <label class="flex items-center">
                        <input type="checkbox" name="force_update" value="1"
                               <?php echo ($editVersion && $editVersion['force_update']) ? 'checked' : ''; ?>
                               class="mr-2 rounded border-gray-300 text-teal-600 focus:ring-teal-500">
                        <span class="text-sm font-medium text-gray-700">Force Update</span>
                    </label>
                    <p class="text-xs text-gray-500 mt-1 ml-6">If enabled, users must update before using the app</p>
                </div>

                <div class="mb-4">
                    <label class="flex items-center">
                        <input type="checkbox" name="is_active" value="1"
                               <?php echo (!$editVersion || $editVersion['is_active']) ? 'checked' : ''; ?>
                               class="mr-2 rounded border-gray-300 text-teal-600 focus:ring-teal-500">
                        <span class="text-sm font-medium text-gray-700">Active</span>
                    </label>
                    <p class="text-xs text-gray-500 mt-1 ml-6">Only one active version should exist. Adding new active version will deactivate others.</p>
                </div>

                <div class="flex justify-end space-x-3">
                    <button type="button" onclick="closeModal()"
                            class="px-4 py-2 border border-gray-300 rounded-md hover:bg-gray-50">
                        Cancel
                    </button>
                    <button type="submit"
                            class="px-4 py-2 bg-teal-600 text-white rounded-md hover:bg-teal-700">
                        <?php echo $editVersion ? 'Update' : 'Add'; ?> Version
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function openAddModal() {
            document.getElementById('versionModal').classList.remove('hidden');
        }

        function closeModal() {
            document.getElementById('versionModal').classList.add('hidden');
            window.location.href = 'version.php';
        }

        <?php if ($editVersion): ?>
            document.getElementById('versionModal').classList.remove('hidden');
        <?php endif; ?>
    </script>

<?php include 'layout_end.php'; ?>

