<?php
/**
 * Dynamic Fields Manager
 */

$pageTitle = 'Fields Manager';
require_once __DIR__ . '/includes/auth_check.php';

$db = getDB();

$selectedTypeId = $_GET['social_type_id'] ?? null;

// Handle actions (BEFORE any output)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add' && $selectedTypeId) {
        $fieldName = trim($_POST['field_name'] ?? '');
        $fieldLabel = trim($_POST['field_label'] ?? '');
        $fieldType = $_POST['field_type'] ?? 'text';
        $fieldOptions = $_POST['field_options'] ?? '';
        $isRequired = isset($_POST['is_required']) ? 1 : 0;
        $placeholder = trim($_POST['placeholder'] ?? '');
        $tooltip = trim($_POST['tooltip'] ?? '');
        $sortOrder = intval($_POST['sort_order'] ?? 0);
        
        if (!empty($fieldName) && !empty($fieldLabel)) {
            $optionsJson = ($fieldType === 'dropdown' && !empty($fieldOptions)) ? json_encode(explode("\n", $fieldOptions)) : null;
            
            $stmt = $db->prepare("
                INSERT INTO social_media_fields 
                (social_type_id, field_name, field_label, field_type, field_options, is_required, placeholder, tooltip, sort_order)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$selectedTypeId, $fieldName, $fieldLabel, $fieldType, $optionsJson, $isRequired, $placeholder, $tooltip, $sortOrder]);
        }
    } elseif ($action === 'edit' && isset($_POST['id'])) {
        $id = intval($_POST['id']);
        $fieldName = trim($_POST['field_name'] ?? '');
        $fieldLabel = trim($_POST['field_label'] ?? '');
        $fieldType = $_POST['field_type'] ?? 'text';
        $fieldOptions = $_POST['field_options'] ?? '';
        $isRequired = isset($_POST['is_required']) ? 1 : 0;
        $placeholder = trim($_POST['placeholder'] ?? '');
        $tooltip = trim($_POST['tooltip'] ?? '');
        $sortOrder = intval($_POST['sort_order'] ?? 0);
        
        $optionsJson = ($fieldType === 'dropdown' && !empty($fieldOptions)) ? json_encode(explode("\n", $fieldOptions)) : null;
        
        $stmt = $db->prepare("
            UPDATE social_media_fields 
            SET field_name = ?, field_label = ?, field_type = ?, field_options = ?, is_required = ?, placeholder = ?, tooltip = ?, sort_order = ?
            WHERE id = ?
        ");
        $stmt->execute([$fieldName, $fieldLabel, $fieldType, $optionsJson, $isRequired, $placeholder, $tooltip, $sortOrder, $id]);
    } elseif ($action === 'delete' && isset($_POST['id'])) {
        $id = intval($_POST['id']);
        $stmt = $db->prepare("DELETE FROM social_media_fields WHERE id = ?");
        $stmt->execute([$id]);
    } elseif ($action === 'reorder' && isset($_POST['order'])) {
        $order = json_decode($_POST['order'], true);
        foreach ($order as $index => $fieldId) {
            $stmt = $db->prepare("UPDATE social_media_fields SET sort_order = ? WHERE id = ?");
            $stmt->execute([$index, $fieldId]);
        }
        echo json_encode(['success' => true]);
        exit;
    }
    
    header('Location: fields_manager.php?social_type_id=' . $selectedTypeId);
    exit;
}

// Now include header (after POST handling)
require_once __DIR__ . '/includes/header.php';

// Get all social types
$stmt = $db->query("SELECT id, name FROM social_media_types ORDER BY name");
$socialTypes = $stmt->fetchAll();

// Get fields for selected type
$fields = [];
if ($selectedTypeId) {
    $stmt = $db->prepare("
        SELECT * FROM social_media_fields 
        WHERE social_type_id = ? 
        ORDER BY sort_order ASC, id ASC
    ");
    $stmt->execute([$selectedTypeId]);
    $fields = $stmt->fetchAll();
    
    // Parse dropdown options
    foreach ($fields as &$field) {
        if ($field['field_type'] === 'dropdown' && !empty($field['field_options'])) {
            $field['field_options_array'] = json_decode($field['field_options'], true);
        }
    }
}
?>

<div class="bg-white rounded-lg shadow mb-6">
    <div class="p-6 border-b">
        <h2 class="text-xl font-bold mb-4">Fields Manager</h2>
        
        <!-- Social Type Selector -->
        <div class="flex items-center space-x-4">
            <label class="font-medium">Select Social Media Type:</label>
            <select id="typeSelector" onchange="window.location.href='?social_type_id=' + this.value" class="border rounded px-3 py-2">
                <option value="">-- Select Type --</option>
                <?php foreach ($socialTypes as $type): ?>
                    <option value="<?php echo $type['id']; ?>" <?php echo $selectedTypeId == $type['id'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($type['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
    </div>
    
    <?php if ($selectedTypeId): ?>
    <div class="p-6">
        <div class="flex justify-between items-center mb-4">
            <h3 class="text-lg font-semibold">Fields for <?php echo htmlspecialchars($socialTypes[array_search($selectedTypeId, array_column($socialTypes, 'id'))]['name'] ?? ''); ?></h3>
            <button onclick="showAddModal()" class="bg-blue-500 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded">
                <i class="fas fa-plus"></i> Add Field
            </button>
        </div>
        
        <div id="fieldsList" class="space-y-2">
            <?php foreach ($fields as $field): ?>
            <div class="border rounded p-4 flex items-center justify-between" data-field-id="<?php echo $field['id']; ?>">
                <div class="flex-1">
                    <div class="flex items-center space-x-4">
                        <i class="fas fa-grip-vertical text-gray-400 cursor-move"></i>
                        <span class="font-medium"><?php echo htmlspecialchars($field['field_label']); ?></span>
                        <span class="text-sm text-gray-500">(<?php echo htmlspecialchars($field['field_name']); ?>)</span>
                        <span class="px-2 py-1 text-xs rounded bg-blue-100 text-blue-800"><?php echo htmlspecialchars($field['field_type']); ?></span>
                        <?php if ($field['is_required']): ?>
                            <span class="px-2 py-1 text-xs rounded bg-red-100 text-red-800">Required</span>
                        <?php endif; ?>
                    </div>
                    <?php if ($field['placeholder']): ?>
                        <div class="text-sm text-gray-500 mt-1">Placeholder: <?php echo htmlspecialchars($field['placeholder']); ?></div>
                    <?php endif; ?>
                </div>
                <div class="flex space-x-2">
                    <button onclick="editField(<?php echo htmlspecialchars(json_encode($field)); ?>)" class="text-blue-600 hover:text-blue-800">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button onclick="deleteField(<?php echo $field['id']; ?>)" class="text-red-600 hover:text-red-800">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
            <?php endforeach; ?>
            
            <?php if (empty($fields)): ?>
                <p class="text-gray-500 text-center py-8">No fields configured yet. Click "Add Field" to get started.</p>
            <?php endif; ?>
        </div>
    </div>
    <?php else: ?>
    <div class="p-6 text-center text-gray-500">
        Please select a social media type to manage its fields.
    </div>
    <?php endif; ?>
</div>

<!-- Add/Edit Modal -->
<div id="fieldModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded-lg p-6 w-full max-w-md max-h-[90vh] overflow-y-auto">
        <h3 class="text-xl font-bold mb-4" id="modalTitle">Add Field</h3>
        <form method="POST" id="fieldForm">
            <input type="hidden" name="action" id="formAction" value="add">
            <input type="hidden" name="id" id="formId">
            
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">Field Name (Internal)</label>
                <input type="text" name="field_name" id="formFieldName" required class="w-full border rounded px-3 py-2" placeholder="e.g., username">
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">Field Label (Display)</label>
                <input type="text" name="field_label" id="formFieldLabel" required class="w-full border rounded px-3 py-2" placeholder="e.g., Username/Email">
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">Field Type</label>
                <select name="field_type" id="formFieldType" required onchange="toggleOptions()" class="w-full border rounded px-3 py-2">
                    <option value="text">Text</option>
                    <option value="number">Number</option>
                    <option value="password">Password</option>
                    <option value="textarea">Textarea</option>
                    <option value="dropdown">Dropdown</option>
                    <option value="checkbox">Checkbox</option>
                    <option value="file">File Upload</option>
                </select>
            </div>
            
            <div class="mb-4" id="optionsContainer" style="display: none;">
                <label class="block text-sm font-medium mb-2">Options (one per line)</label>
                <textarea name="field_options" id="formFieldOptions" class="w-full border rounded px-3 py-2" rows="4" placeholder="Option 1&#10;Option 2&#10;Option 3"></textarea>
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">Placeholder</label>
                <input type="text" name="placeholder" id="formPlaceholder" class="w-full border rounded px-3 py-2">
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">Tooltip</label>
                <textarea name="tooltip" id="formTooltip" class="w-full border rounded px-3 py-2" rows="2"></textarea>
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">Sort Order</label>
                <input type="number" name="sort_order" id="formSortOrder" value="0" class="w-full border rounded px-3 py-2">
            </div>
            
            <div class="mb-4">
                <label class="flex items-center">
                    <input type="checkbox" name="is_required" id="formIsRequired" class="mr-2">
                    <span>Required Field</span>
                </label>
            </div>
            
            <div class="flex justify-end space-x-2">
                <button type="button" onclick="closeModal()" class="px-4 py-2 border rounded">Cancel</button>
                <button type="submit" class="px-4 py-2 bg-blue-500 text-white rounded">Save</button>
            </div>
        </form>
    </div>
</div>

<script>
function showAddModal() {
    document.getElementById('modalTitle').textContent = 'Add Field';
    document.getElementById('formAction').value = 'add';
    document.getElementById('fieldForm').reset();
    document.getElementById('formId').value = '';
    toggleOptions();
    document.getElementById('fieldModal').classList.remove('hidden');
}

function editField(field) {
    document.getElementById('modalTitle').textContent = 'Edit Field';
    document.getElementById('formAction').value = 'edit';
    document.getElementById('formId').value = field.id;
    document.getElementById('formFieldName').value = field.field_name;
    document.getElementById('formFieldLabel').value = field.field_label;
    document.getElementById('formFieldType').value = field.field_type;
    document.getElementById('formPlaceholder').value = field.placeholder || '';
    document.getElementById('formTooltip').value = field.tooltip || '';
    document.getElementById('formSortOrder').value = field.sort_order;
    document.getElementById('formIsRequired').checked = field.is_required == 1;
    
    if (field.field_type === 'dropdown' && field.field_options_array) {
        document.getElementById('formFieldOptions').value = field.field_options_array.join('\n');
    }
    
    toggleOptions();
    document.getElementById('fieldModal').classList.remove('hidden');
}

function deleteField(id) {
    if (confirm('Are you sure you want to delete this field?')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = '<input type="hidden" name="action" value="delete"><input type="hidden" name="id" value="' + id + '">';
        document.body.appendChild(form);
        form.submit();
    }
}

function toggleOptions() {
    const fieldType = document.getElementById('formFieldType').value;
    const container = document.getElementById('optionsContainer');
    container.style.display = fieldType === 'dropdown' ? 'block' : 'none';
}

function closeModal() {
    document.getElementById('fieldModal').classList.add('hidden');
}
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>

