<?php
/**
 * Manual Deposit Gateways Management
 */

$pageTitle = 'Manual Deposit Gateways';
require_once __DIR__ . '/includes/auth_check.php';

$db = getDB();

// Handle actions (BEFORE any output)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add') {
        $name = trim($_POST['name'] ?? '');
        $logo = trim($_POST['logo'] ?? '');
        $paymentType = $_POST['payment_type'] ?? 'personal';
        $paymentNumber = trim($_POST['payment_number'] ?? '');
        $accountName = trim($_POST['account_name'] ?? '');
        $instructions = $_POST['instructions'] ?? '';
        $minAmount = floatval($_POST['min_amount'] ?? 0);
        $maxAmount = floatval($_POST['max_amount'] ?? 10000);
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        $sortOrder = intval($_POST['sort_order'] ?? 0);
        
        if (!empty($name) && !empty($paymentNumber)) {
            $stmt = $db->prepare("
                INSERT INTO manual_gateways (name, logo, payment_type, payment_number, account_name, instructions, min_amount, max_amount, is_active, sort_order)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$name, $logo, $paymentType, $paymentNumber, $accountName, $instructions, $minAmount, $maxAmount, $isActive, $sortOrder]);
        }
    } elseif ($action === 'edit' && isset($_POST['id'])) {
        $id = intval($_POST['id']);
        $name = trim($_POST['name'] ?? '');
        $logo = trim($_POST['logo'] ?? '');
        $paymentType = $_POST['payment_type'] ?? 'personal';
        $paymentNumber = trim($_POST['payment_number'] ?? '');
        $accountName = trim($_POST['account_name'] ?? '');
        $instructions = $_POST['instructions'] ?? '';
        $minAmount = floatval($_POST['min_amount'] ?? 0);
        $maxAmount = floatval($_POST['max_amount'] ?? 10000);
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        $sortOrder = intval($_POST['sort_order'] ?? 0);
        
        $stmt = $db->prepare("
            UPDATE manual_gateways 
            SET name = ?, logo = ?, payment_type = ?, payment_number = ?, account_name = ?, instructions = ?, min_amount = ?, max_amount = ?, is_active = ?, sort_order = ?
            WHERE id = ?
        ");
        $stmt->execute([$name, $logo, $paymentType, $paymentNumber, $accountName, $instructions, $minAmount, $maxAmount, $isActive, $sortOrder, $id]);
    } elseif ($action === 'delete' && isset($_POST['id'])) {
        $id = intval($_POST['id']);
        $stmt = $db->prepare("DELETE FROM manual_gateways WHERE id = ?");
        $stmt->execute([$id]);
    }
    
    header('Location: manual_gateways.php');
    exit;
}

// Now include header (after POST handling)
require_once __DIR__ . '/includes/header.php';

// Get all gateways
$stmt = $db->query("SELECT * FROM manual_gateways ORDER BY sort_order, name ASC");
$gateways = $stmt->fetchAll();
?>

<div class="bg-white rounded-lg shadow mb-6">
    <div class="p-6 border-b flex justify-between items-center">
        <h2 class="text-xl font-bold">Manual Deposit Gateways</h2>
        <button onclick="showAddModal()" class="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600">
            <i class="fas fa-plus"></i> Add Gateway
        </button>
    </div>
    
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Name</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Type</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Payment Number</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Min/Max</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($gateways as $gateway): ?>
                <tr>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div class="flex items-center">
                            <?php if ($gateway['logo']): ?>
                                <img src="<?php echo htmlspecialchars($gateway['logo']); ?>" alt="" class="w-8 h-8 rounded mr-2">
                            <?php endif; ?>
                            <div class="font-medium"><?php echo htmlspecialchars($gateway['name']); ?></div>
                        </div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="px-2 py-1 text-xs rounded bg-gray-100">
                            <?php echo ucfirst($gateway['payment_type']); ?>
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm"><?php echo htmlspecialchars($gateway['payment_number']); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm">
                        ৳<?php echo number_format($gateway['min_amount'], 2); ?> - ৳<?php echo number_format($gateway['max_amount'], 2); ?>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="px-2 py-1 text-xs rounded-full <?php echo $gateway['is_active'] ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                            <?php echo $gateway['is_active'] ? 'Active' : 'Inactive'; ?>
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm">
                        <button onclick="editGateway(<?php echo htmlspecialchars(json_encode($gateway)); ?>)" class="text-blue-600 hover:text-blue-800 mr-3">
                            <i class="fas fa-edit"></i> Edit
                        </button>
                        <button onclick="deleteGateway(<?php echo $gateway['id']; ?>)" class="text-red-600 hover:text-red-800">
                            <i class="fas fa-trash"></i> Delete
                        </button>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Add/Edit Modal -->
<div id="gatewayModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded-lg p-6 w-full max-w-2xl max-h-[90vh] overflow-y-auto">
        <h3 class="text-xl font-bold mb-4" id="modalTitle">Add Gateway</h3>
        <form method="POST" id="gatewayForm">
            <input type="hidden" name="action" id="formAction" value="add">
            <input type="hidden" name="id" id="formId">
            
            <div class="space-y-4">
                <div>
                    <label class="block text-sm font-medium mb-2">Gateway Name *</label>
                    <input type="text" name="name" id="formName" required class="w-full border rounded px-3 py-2">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2">Logo URL</label>
                    <input type="text" name="logo" id="formLogo" class="w-full border rounded px-3 py-2" placeholder="https://example.com/logo.png">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2">Payment Type *</label>
                    <select name="payment_type" id="formPaymentType" required class="w-full border rounded px-3 py-2">
                        <option value="personal">Personal</option>
                        <option value="business">Business</option>
                    </select>
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2">Payment Number/Account *</label>
                    <input type="text" name="payment_number" id="formPaymentNumber" required class="w-full border rounded px-3 py-2" placeholder="01XXXXXXXXX">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2">Account Name</label>
                    <input type="text" name="account_name" id="formAccountName" class="w-full border rounded px-3 py-2" placeholder="Account holder name">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2">Instructions (HTML allowed)</label>
                    <textarea name="instructions" id="formInstructions" rows="4" class="w-full border rounded px-3 py-2" placeholder="Payment instructions..."></textarea>
                </div>
                
                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium mb-2">Min Amount</label>
                        <input type="number" name="min_amount" id="formMinAmount" step="0.01" value="0" class="w-full border rounded px-3 py-2">
                    </div>
                    <div>
                        <label class="block text-sm font-medium mb-2">Max Amount</label>
                        <input type="number" name="max_amount" id="formMaxAmount" step="0.01" value="10000" class="w-full border rounded px-3 py-2">
                    </div>
                </div>
                
                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium mb-2">Sort Order</label>
                        <input type="number" name="sort_order" id="formSortOrder" value="0" class="w-full border rounded px-3 py-2">
                    </div>
                    <div class="flex items-center">
                        <label class="flex items-center">
                            <input type="checkbox" name="is_active" id="formIsActive" value="1" checked class="mr-2">
                            <span>Active</span>
                        </label>
                    </div>
                </div>
            </div>
            
            <div class="flex justify-end space-x-2 mt-6">
                <button type="button" onclick="closeModal()" class="px-4 py-2 border rounded">Cancel</button>
                <button type="submit" class="px-4 py-2 bg-blue-500 text-white rounded">Save</button>
            </div>
        </form>
    </div>
</div>

<script>
function showAddModal() {
    document.getElementById('modalTitle').textContent = 'Add Gateway';
    document.getElementById('formAction').value = 'add';
    document.getElementById('gatewayForm').reset();
    document.getElementById('formId').value = '';
    document.getElementById('formIsActive').checked = true;
    document.getElementById('gatewayModal').classList.remove('hidden');
}

function editGateway(gateway) {
    document.getElementById('modalTitle').textContent = 'Edit Gateway';
    document.getElementById('formAction').value = 'edit';
    document.getElementById('formId').value = gateway.id;
    document.getElementById('formName').value = gateway.name || '';
    document.getElementById('formLogo').value = gateway.logo || '';
    document.getElementById('formPaymentType').value = gateway.payment_type || 'personal';
    document.getElementById('formPaymentNumber').value = gateway.payment_number || '';
    document.getElementById('formAccountName').value = gateway.account_name || '';
    document.getElementById('formInstructions').value = gateway.instructions || '';
    document.getElementById('formMinAmount').value = gateway.min_amount || 0;
    document.getElementById('formMaxAmount').value = gateway.max_amount || 10000;
    document.getElementById('formSortOrder').value = gateway.sort_order || 0;
    document.getElementById('formIsActive').checked = gateway.is_active == 1;
    document.getElementById('gatewayModal').classList.remove('hidden');
}

function deleteGateway(id) {
    if (confirm('Are you sure you want to delete this gateway?')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <input type="hidden" name="action" value="delete">
            <input type="hidden" name="id" value="${id}">
        `;
        document.body.appendChild(form);
        form.submit();
    }
}

function closeModal() {
    document.getElementById('gatewayModal').classList.add('hidden');
}
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>

