<?php
/**
 * Wallet API
 * Deposits, withdrawals, transactions
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/middleware.php';
require_once __DIR__ . '/../database.php';

$db = getDB();
$method = $_SERVER['REQUEST_METHOD'];

// Get wallet balance and transactions
if ($method === 'GET') {
    $user = requireAuth();
    
    $stmt = $db->prepare("SELECT wallet_balance FROM users WHERE id = ?");
    $stmt->execute([$user['id']]);
    $userData = $stmt->fetch();
    
    // Get transactions
    $page = $_GET['page'] ?? 1;
    $limit = 20;
    $offset = ($page - 1) * $limit;
    
    $stmt = $db->prepare("
        SELECT * FROM transactions 
        WHERE user_id = ? 
        ORDER BY created_at DESC 
        LIMIT ? OFFSET ?
    ");
    $stmt->execute([$user['id'], $limit, $offset]);
    $transactions = $stmt->fetchAll();
    
    // Get total count
    $stmt = $db->prepare("SELECT COUNT(*) as total FROM transactions WHERE user_id = ?");
    $stmt->execute([$user['id']]);
    $total = $stmt->fetch()['total'];
    
    echo json_encode([
        'success' => true,
        'data' => [
            'balance' => (float)$userData['wallet_balance'],
            'transactions' => $transactions,
            'pagination' => [
                'page' => (int)$page,
                'limit' => $limit,
                'total' => (int)$total,
                'pages' => ceil($total / $limit)
            ]
        ]
    ]);
    exit;
}

// Create deposit request
if ($method === 'POST' && isset($_GET['action']) && $_GET['action'] === 'deposit') {
    $user = requireAuth();
    $input = json_decode(file_get_contents('php://input'), true);
    
    $amount = $input['amount'] ?? null;
    $gateway = $input['gateway'] ?? '';
    $gatewayType = $input['gateway_type'] ?? 'manual';
    $transactionId = $input['transaction_id'] ?? null;
    $senderNumber = $input['sender_number'] ?? null;
    $screenshot = $input['screenshot'] ?? null;
    
    if (empty($amount) || $amount <= 0 || empty($gateway)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Invalid deposit data']);
        exit;
    }
    
    // Validate gateway limits if manual
    if ($gatewayType === 'manual') {
        $stmt = $db->prepare("SELECT min_amount, max_amount FROM manual_gateways WHERE name = ? AND is_active = 1");
        $stmt->execute([$gateway]);
        $gatewayData = $stmt->fetch();
        
        if (!$gatewayData) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid gateway']);
            exit;
        }
        
        if ($amount < $gatewayData['min_amount'] || $amount > $gatewayData['max_amount']) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => "Amount must be between {$gatewayData['min_amount']} and {$gatewayData['max_amount']}"]);
            exit;
        }
    }
    
    $stmt = $db->prepare("
        INSERT INTO deposits (user_id, amount, gateway, gateway_type, transaction_id, sender_number, screenshot, status)
        VALUES (?, ?, ?, ?, ?, ?, ?, 'pending')
    ");
    $stmt->execute([$user['id'], $amount, $gateway, $gatewayType, $transactionId, $senderNumber, $screenshot]);
    
    $depositId = $db->lastInsertId();
    
    echo json_encode([
        'success' => true,
        'message' => 'Deposit request submitted',
        'data' => ['deposit_id' => $depositId]
    ]);
    exit;
}

// Get deposit requests
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'deposits') {
    $user = requireAuth();
    
    $stmt = $db->prepare("
        SELECT * FROM deposits 
        WHERE user_id = ? 
        ORDER BY created_at DESC
    ");
    $stmt->execute([$user['id']]);
    $deposits = $stmt->fetchAll();
    
    echo json_encode(['success' => true, 'data' => $deposits]);
    exit;
}

// Create withdraw request
if ($method === 'POST' && isset($_GET['action']) && $_GET['action'] === 'withdraw') {
    $user = requireAuth();
    $input = json_decode(file_get_contents('php://input'), true);
    
    $amount = $input['amount'] ?? null;
    $gatewayId = $input['gateway_id'] ?? null;
    $accountNumber = $input['account_number'] ?? '';
    $accountName = $input['account_name'] ?? '';
    
    if (empty($amount) || $amount <= 0 || empty($gatewayId) || empty($accountNumber)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Invalid withdraw data']);
        exit;
    }
    
    // Check wallet balance
    $stmt = $db->prepare("SELECT wallet_balance FROM users WHERE id = ?");
    $stmt->execute([$user['id']]);
    $userData = $stmt->fetch();
    
    if ($userData['wallet_balance'] < $amount) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Insufficient balance']);
        exit;
    }
    
    // Validate gateway
    $stmt = $db->prepare("SELECT min_amount, max_amount FROM withdraw_gateways WHERE id = ? AND is_active = 1");
    $stmt->execute([$gatewayId]);
    $gateway = $stmt->fetch();
    
    if (!$gateway) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Invalid gateway']);
        exit;
    }
    
    if ($amount < $gateway['min_amount'] || $amount > $gateway['max_amount']) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => "Amount must be between {$gateway['min_amount']} and {$gateway['max_amount']}"]);
        exit;
    }
    
    // Deduct from wallet
    $newBalance = $userData['wallet_balance'] - $amount;
    $stmt = $db->prepare("UPDATE users SET wallet_balance = ? WHERE id = ?");
    $stmt->execute([$newBalance, $user['id']]);
    
    // Record transaction
    $stmt = $db->prepare("
        INSERT INTO transactions (user_id, type, amount, balance_before, balance_after, description)
        VALUES (?, 'withdraw', ?, ?, ?, ?)
    ");
    $stmt->execute([$user['id'], -$amount, $userData['wallet_balance'], $newBalance, "Withdraw request"]);
    
    // Create withdraw request
    $stmt = $db->prepare("
        INSERT INTO withdraw_requests (user_id, amount, gateway_id, account_number, account_name, status)
        VALUES (?, ?, ?, ?, ?, 'pending')
    ");
    $stmt->execute([$user['id'], $amount, $gatewayId, $accountNumber, $accountName]);
    
    $withdrawId = $db->lastInsertId();
    
    echo json_encode([
        'success' => true,
        'message' => 'Withdraw request submitted',
        'data' => ['withdraw_id' => $withdrawId]
    ]);
    exit;
}

// Get withdraw requests
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'withdraws') {
    $user = requireAuth();
    
    $stmt = $db->prepare("
        SELECT wr.*, wg.name as gateway_name, wg.type as gateway_type
        FROM withdraw_requests wr
        JOIN withdraw_gateways wg ON wr.gateway_id = wg.id
        WHERE wr.user_id = ? 
        ORDER BY wr.created_at DESC
    ");
    $stmt->execute([$user['id']]);
    $withdraws = $stmt->fetchAll();
    
    echo json_encode(['success' => true, 'data' => $withdraws]);
    exit;
}

// Get available gateways
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'gateways') {
    $type = $_GET['type'] ?? 'deposit'; // deposit or withdraw
    
    if ($type === 'deposit') {
        // Get manual deposit gateways
        $stmt = $db->query("SELECT id, name, logo, payment_type, min_amount, max_amount, instructions FROM manual_gateways WHERE is_active = 1 ORDER BY sort_order");
        $gateways = $stmt->fetchAll();
        
        // Get automatic gateways from settings
        $stmt = $db->query("SELECT setting_key, setting_value FROM settings WHERE setting_key LIKE '%_enabled' AND group_name = 'payment'");
        $settings = $stmt->fetchAll();
        $autoGateways = [];
        foreach ($settings as $setting) {
            if ($setting['setting_value'] == '1') {
                $gatewayName = str_replace('_enabled', '', $setting['setting_key']);
                $autoGateways[] = [
                    'name' => ucfirst($gatewayName),
                    'type' => 'automatic',
                    'slug' => $gatewayName
                ];
            }
        }
        
        echo json_encode([
            'success' => true,
            'data' => [
                'manual' => $gateways,
                'automatic' => $autoGateways
            ]
        ]);
    } else {
        // Get withdraw gateways
        $stmt = $db->query("SELECT id, name, type, logo, min_amount, max_amount, instruction FROM withdraw_gateways WHERE is_active = 1");
        $gateways = $stmt->fetchAll();
        
        echo json_encode(['success' => true, 'data' => $gateways]);
    }
    exit;
}

http_response_code(405);
echo json_encode(['success' => false, 'message' => 'Method not allowed']);

